import json
import logging

import diff_match_patch
import pandas as pd
from flask import Markup, request, url_for, flash, current_app as app
from flask_admin import expose
from flask_admin.babel import lazy_gettext, gettext
from flask_admin.contrib.mongoengine.filters import BaseMongoEngineFilter
from flask_admin.contrib.mongoengine.helpers import format_error
from flask_admin.form.upload import FileUploadField
from flask_admin.model.fields import AjaxSelectField, AjaxSelectMultipleField
from flask_wtf import Form
from wtforms import StringField, IntegerField, SelectMultipleField
from wtforms.validators import DataRequired, Length, Regexp, ValidationError, StopValidation

from jafar import db, localization_mongo
from jafar.admin import beta
from jafar.admin.ajax import TranslationsAjaxLoader, CountryAjaxLoader, GeoCodesAjaxModelLoader
from jafar.admin.beta import SetupWizardBetaStorage
from jafar.admin.fields import MdsFileUploadField, launcher_mds_client, PrettyJSONField
from jafar.admin.model_converter import CustomModelConverter
from jafar.admin.models.base import ROLE_ADMIN, ROLE_DEVELOPER, ROLE_TESTER, ROLE_RESOURCE, ROLE_SUPPORT
from jafar.admin.validators import ListUniqueValidator
from jafar.admin.views.base import ModelView, DEV_USERNAME, get_s3_image_subdocument_config, AuthAdminView
from jafar.models.yandexphone import GiftSet, Phone, ImeiField, ALL_GIFTS, PLUS_ID, DISK_ID

logger = logging.getLogger(__name__)

MAC_REGEXP = r'^(?:[0-9a-fA-F]{2}:?){5}[0-9a-fA-F]{2}$'


class AdministrationView(ModelView):
    access_roles = (ROLE_ADMIN,)


class HistoryView(ModelView):
    can_create = False
    can_edit = False
    can_delete = False

    # sort by timestamp in descending order
    column_default_sort = ('timestamp', True)  # Descending
    column_list = ('timestamp', 'user', 'action', 'diff', 'document_class')

    def _format_username(self, context, model, name):
        if model.user == DEV_USERNAME:
            return DEV_USERNAME
        return Markup('<span style="white-space:nowrap;">'
                      '<img src="//center.yandex-team.ru/api/v1/user/{0}/avatar/20.jpg" alt="">'
                      ' <a href="//staff.yandex-team.ru/{0}"><b>{0}<b></a>'
                      '</span>').format(model.user)

    def _format_diff(self, context, model, name):
        def htmlize(data):
            lines = []
            for line in data.split('\n'):
                line = line.replace("&", "&amp;").replace("<", "&lt;").replace(">", "&gt;")
                indent = len(line) - len(line.lstrip())
                line = '&nbsp;' * indent + line
                lines.append(line)
            return '<br>'.join(lines)

        def prettify_json(json_data):
            return json.dumps(json.loads(json_data), encoding='utf-8', ensure_ascii=False)

        del_template = "<del style=\"background:#ffe6e6;\">%s</del>"
        insert_template = "<ins style=\"background:#e6ffe6;\">%s</ins>"
        equal_template = "<span>%s</span>"

        if model.document_before and model.document_after:
            before = prettify_json(model.document_before)
            after = prettify_json(model.document_after)
            dmp = diff_match_patch.diff_match_patch()
            diffs = dmp.diff_main(before, after)
            dmp.diff_cleanupSemantic(diffs)

            html = []
            for (op, data) in diffs:
                text = htmlize(data)
                if op == dmp.DIFF_INSERT:
                    html.append(insert_template % text)
                elif op == dmp.DIFF_DELETE:
                    html.append(del_template % text)
                elif op == dmp.DIFF_EQUAL:
                    html.append(equal_template % text)
            return Markup("".join(html))
        elif model.document_after:
            text = htmlize(prettify_json(model.document_after))
            return Markup(insert_template % text)
        elif model.document_before:
            text = htmlize(prettify_json(model.document_before))
            return Markup(del_template % text)

    column_formatters = {
        'user': _format_username,
        'diff': _format_diff,
    }


class ExperimentModelView(ModelView):
    def scaffold_list_columns(self):
        return super(ExperimentModelView, self).scaffold_list_columns() + ['is_used']


class RecommenderModelView(ModelView):
    # mongoengine ajax loader should have such formed title to be successfully exposed
    loader = TranslationsAjaxLoader('title_pairs-title')
    form_subdocuments = {
        'title_pairs': {
            'form_subdocuments': {
                # None is used here to access the forms embedded into ListField
                None: {
                    'form_columns': ('title', 'subtitle'),
                    'form_ajax_refs': {'title': loader, 'subtitle': loader},
                    'form_extra_fields': {'title': AjaxSelectField(loader, 'Title'),
                                          'subtitle': AjaxSelectField(loader, 'Subtitle')}
                }
            }
        }
    }

    def scaffold_list_columns(self):
        return super(RecommenderModelView, self).scaffold_list_columns() + ['is_used']


class BlacklistModelView(ModelView):
    country_loader = CountryAjaxLoader('targeting-countries', placeholder='Start typing country alpha-2 code')

    column_list = ('name', 'description', 'component', 'targeting', 'package_names', 'package_names_regex',
                   'categories', 'white_list', 'publishers', 'is_launcher')

    form_subdocuments = {
        'targeting': {
            'form_ajax_refs': {'countries': country_loader},
            'form_extra_fields': {'countries': AjaxSelectMultipleField(country_loader, 'Countries')}
        }
    }


class StrategyModelView(ModelView):
    column_list = ('name', 'description', 'constraints', 'card_configs', 'randomization')

    def _format_card_configs(self, context, model, name):
        if not model.card_configs:
            return ''
        return Markup('<pre>{}</pre>'.format(json.dumps({
            item.card_config.__class__.__name__: item.weight
            for item in model.card_configs
        }, indent=2)))

    column_formatters = {
        'card_configs': _format_card_configs,
    }


class AppView(ModelView):
    column_filters = ('package_name',)
    form_overrides = {
        'short_titles': PrettyJSONField,
        'short_titles_override': PrettyJSONField,
    }

    def _format_short_titles(self, context, model, name):
        titles = model[name]
        return Markup(u'<br>'.join(u'%s: %s' % (lang, titles[lang]) for lang in sorted(titles)))

    column_formatters = {
        'short_titles': _format_short_titles,
        'short_titles_override': _format_short_titles,
    }


class BetaStorageUploadField(FileUploadField):
    def __init__(self, **kwargs):
        super(BetaStorageUploadField, self).__init__(allow_overwrite=True, **kwargs)

    @classmethod
    def upload_to_beta(cls, package_name, data):
        status = SetupWizardBetaStorage.upload(package_name=package_name, fp=data.read())
        if status != 200:
            raise RuntimeError('cant upload to beta, return code %s', status)

        return SetupWizardBetaStorage.get_download_url(package_name)

    def _get_path(self, package_name):
        return package_name

    def _delete_file(self, package_name):
        pass

    def _save_file(self, data, package_name):
        url = self.upload_to_beta(package_name, data)
        try:
            beta.check_url(url)
        except beta.AppNotFound as ex:
            logger.warning(ex)
        return url


def get_mds_icon_filename(model, data):
    package_name = model.package_name
    yphone_dpi = 'xxxhdpi'
    return '{package_name}-{size}'.format(package_name=package_name, size=yphone_dpi)


def get_package_name(model, data):
    return model.package_name


def format_image(view, context, model, name):
    image = getattr(model, name, None)
    if image:
        return Markup('<span style="white-space:nowrap;">'
                      ' <img src="{0}"/ height="60px">'
                      '</span>').format(launcher_mds_client.read_url(image))


def format_s3_image(view, context, model, name):
    image = getattr(model, name, None)
    if image:
        return Markup('<span style="white-space:nowrap;"><img src="{0}"/ height="60px"></span>').format(image.url)


class YandexDistributedAppView(ModelView):
    access_roles = ROLE_ADMIN, ROLE_DEVELOPER, ROLE_RESOURCE

    def _format_download_url(self, context, model, name):
        if not model.download_url:
            return

        return Markup('<span style="white-space:nowrap;">'
                      ' <a href="{0}">{1}</a>'
                      '</span>').format(model.download_url, model.package_name)

    def _format_description(self, context, model, name):
        text = model.get_text('ru')
        if text:
            return Markup(
                '<span style="white-space:nowrap;"><b>{0.title}</b></span>'
                '<br>'
                '<span>{0.description}</span>'
            ).format(text)

    form_overrides = {
        'download_url': BetaStorageUploadField,
        'icon': MdsFileUploadField,
    }

    column_list = ('icon', 'download_url', 'offer_id', 'show_in_feed', 'mark_as_sponsored', 'description',
                   'tag', 'tag_extra', 'content_rating', 'rank', 'regions')

    form_ajax_refs = {
        'regions': GeoCodesAjaxModelLoader('regions'),
    }

    form_extra_fields = {
        'regions': AjaxSelectMultipleField(GeoCodesAjaxModelLoader('regions'), 'Regions'),
    }

    column_formatters = {
        'description': _format_description,
        'download_url': _format_download_url,
        'icon': format_image,
    }

    form_args = {
        'icon': {
            'namegen': get_mds_icon_filename,
            'kind': 'appinstaller_icon',
        },
        'download_url': {
            'namegen': get_package_name,
            'label': 'Apk',
        }
    }

    form_widget_args = {
        'offer_id': {
            'data-role': 'select2-ajax-offer-id',
            'package-name-field': 'package_name',
        },
    }

    def on_model_delete(self, model):
        mds_key = model.icon
        launcher_mds_client.delete(mds_key)


def get_image_name(field):
    return lambda model, data: '{model.id}-{field}'.format(field=field, model=model)


colorwiz_attributes = [
    ('Card background', 'card_background'),
    ('Card text', 'card_text'),
    ('Button background', 'button_background'),
    ('Button text', 'button_text'),
]

colorwiz_template = '<div class="colorwiz"><span>{name}</span><div class="color-box" ' \
                    'style="background-color: {color};"></div></div>'


def format_colorwiz(view, context, model, name):
    colorwiz = getattr(model, name, None)
    colors = (
        colorwiz_template.format(color=colorwiz.get(key), name=name)
        for name, key in colorwiz_attributes
    )
    return Markup(''.join(colors))


class GiftsView(ModelView):
    form_overrides = {
        'suw_preview_image': MdsFileUploadField
    }
    form_args = {
        'suw_preview_image': {
            'namegen': get_image_name('suw_preview_image'),
            'kind': 'gift_image',
        }
    }
    column_formatters = {'suw_preview_image': format_image}
    form_excluded_columns = ('updated_at',)

    column_list = ('id', 'package_name', 'preview_order', 'suw_preview_image')
    column_default_sort = 'preview_order'
    access_roles = ROLE_ADMIN, ROLE_DEVELOPER, ROLE_RESOURCE

    def on_model_delete(self, model):
        mds_key = getattr(model, 'suw_preview_image')
        launcher_mds_client.delete(mds_key)


class BonusCardView(ModelView):
    button_text_key_loader = TranslationsAjaxLoader('button_text_key')

    def _format_texts(self, context, model, name):
        text = model.get_text('ru')
        if text:
            return Markup(
                '<span style="white-space:nowrap;"><b>{0.app_name}</b></span>'
                '<br>'
                '<span style="white-space:nowrap;">{0.title}</span>'
                '<br>'
                '<span>{0.description}</span>'
            ).format(text)

    form_overrides = {
        'banner_image': MdsFileUploadField,
        'icon': MdsFileUploadField,
    }
    form_args = {
        'banner_image': {'namegen': get_image_name('banner_image'), 'kind': 'gift_image'},
        'icon': {'namegen': get_image_name('icon'), 'kind': 'gift_image'},
    }

    column_formatters = {
        'banner_image': format_image,
        'icon': format_image,
        'icon_colorwiz': format_colorwiz,
        'banner_image_colorwiz': format_colorwiz,
        'text': _format_texts,
    }

    form_excluded_columns = ('updated_at',)

    column_list = ('package_name', 'icon', 'banner_image', 'text', 'mark_as_sponsored', 'content_rating',
                   'button_text_key', 'banner_image_colorwiz', 'icon_colorwiz')
    column_default_sort = 'package_name'
    form_ajax_refs = {
        'button_text_key': button_text_key_loader
    }
    form_extra_fields = {
        'button_text_key': AjaxSelectField(button_text_key_loader)
    }
    access_roles = ROLE_ADMIN, ROLE_DEVELOPER, ROLE_RESOURCE

    def on_model_delete(self, model):
        for field in ('banner_image', 'icon'):
            mds_key = getattr(model, field)
            launcher_mds_client.delete(mds_key)


class GiftSetView(ModelView):
    column_list = ('passport_uid', 'gifts', 'phone', 'created_at')
    column_filters = ('passport_uid',)
    form_overrides = {'passport_uid': IntegerField}

    form_widget_args = {
        'passport_uid': {
            'readonly': True,
        },
    }

    access_roles = ROLE_ADMIN, ROLE_DEVELOPER, ROLE_TESTER, ROLE_SUPPORT

    def get_list(self, *args, **kwargs):
        count, gift_sets = super(GiftSetView, self).get_list(*args, **kwargs)
        giftset_to_phone_map = {}
        phones = Phone.objects(gift_set__in=gift_sets).no_dereference().only('id', 'gift_set')
        for phone in phones:
            gift_set_id = phone.gift_set.id
            if gift_set_id in giftset_to_phone_map:
                flash(gettext('Multiple Phone objects for GiftSet: {}'.format(gift_set_id)), category='error')
            giftset_to_phone_map[phone.gift_set.id] = phone.id

        for gift_set in gift_sets:
            gift_set.phone_id = giftset_to_phone_map.get(gift_set.pk)
        return count, gift_sets

    def _format_gifts(self, context, model, name):
        if not model.gifts:
            return ''
        gifts = sorted([given_gift.gift.pk for given_gift in model.gifts])
        return ', '.join(gifts)

    def _format_phone(self, context, model, name):
        phone_id = model.phone_id
        if not phone_id:
            return ''

        return Markup(
            '<a href="{url}" target="_blank">Phone: {phone_id}</a>'.format(
                url=url_for('phone.edit_view', id=phone_id),
                phone_id=phone_id,
            )
        )

    column_formatters = {
        'gifts': _format_gifts,
        'phone': _format_phone,
    }


class FilterGiftSets(BaseMongoEngineFilter):
    def apply(self, query, value):
        return query.filter(gift_set__in=GiftSet.objects(pk=int(value.strip())))

    def operation(self):
        return lazy_gettext('equals')


class PhoneView(ModelView):
    column_list = ('id', 'gift_set', 'device_id', 'imei', 'serial_number', 'wifi_mac', 'batch')
    column_searchable_list = ('id', 'wifi_mac', 'imei', 'imei2', 'serial_number', 'batch')
    column_filters = (FilterGiftSets('gift_set', 'GiftSet ID'),)
    allowed_search_types = (db.StringField, ImeiField)
    access_roles = ROLE_ADMIN, ROLE_DEVELOPER, ROLE_TESTER, ROLE_SUPPORT

    def _format_uuid(self, context, model, name):
        field = model[name]
        if field:
            return Markup('<a href="/admin/user_info/?user_id={0}">{0}</a>').format(field.hex)

    def _format_gift_set(self, context, model, name):
        if not model.gift_set:
            return ''
        gift_set = model.gift_set

        return Markup(
            '<a href="{url}" target="_blank">{gift_set}</a>'.format(
                url=url_for('giftset.edit_view', id=gift_set.pk),
                gift_set=gift_set,
            ) +
            # add form to delete referenced gift_set
            """
            <form class="icon" method="POST" action="/admin/{gift_set_model}/delete/">
                <input id="id" name="id" required="" value="{gift_set_id}" type="hidden">
                <input id="url" name="url" value="/admin/{model}/" type="hidden">
                <button onclick="return safeConfirm('Are you sure you want to delete this record?');" title="Delete record">
                    <span class="fa fa-trash glyphicon glyphicon-trash"></span>
                </button>
            </form>
            """.format(
                model=model._class_name.lower(),
                gift_set_model=gift_set._class_name.lower(),
                gift_set_id=gift_set.pk
            )
        )

    column_formatters = {
        'gift_set': _format_gift_set,
        'device_id': _format_uuid,
    }

    form_overrides = {
        'imei': StringField,
        'imei2': StringField,
    }


class PlusPromocodeView(ModelView):
    form_overrides = {'passport_uid': IntegerField}
    column_searchable_list = ('value',)
    column_filters = ('passport_uid',)

    access_roles = ROLE_ADMIN, ROLE_DEVELOPER, ROLE_TESTER, ROLE_SUPPORT


class GreetingMailView(ModelView):
    column_list = ('passport_uid', 'mail_type', 'plus_promocode', 'phone_id')
    column_searchable_list = ('phone_id', 'plus_promocode')
    column_filters = ('passport_uid', 'mail_type')
    form_overrides = {'passport_uid': IntegerField}

    access_roles = ROLE_ADMIN, ROLE_DEVELOPER, ROLE_TESTER, ROLE_SUPPORT

    def _format_phone(self, context, model, name):
        phone_id = model.phone_id
        if not phone_id:
            return ''

        return Markup(
            '<a href="{url}" target="_blank">{phone_id}</a>'.format(
                url=url_for('phone.edit_view', id=phone_id),
                phone_id=phone_id,
            )
        )

    column_formatters = {
        'phone_id': _format_phone,
    }


class PhoneIdCheckForm(Form):
    wifi_mac = StringField(validators=[DataRequired(), Regexp(MAC_REGEXP)], description="WiFi MAC address")
    bt_mac = StringField(validators=[DataRequired(), Regexp(MAC_REGEXP)], description="Bluetooth MAC address")
    serial_no = StringField(validators=[DataRequired()], description="Serial number")
    imei = StringField(validators=[DataRequired(), Regexp(r'^[0-9]+$'), Length(min=15, max=15)], description="IMEI #1")

    class Meta:
        csrf = False


class PhoneIdCheckView(AuthAdminView):
    @expose('/')
    def index(self, *args, **kwargs):
        form = PhoneIdCheckForm(request.args)
        phone_id = None
        if request.args:
            form.validate()
            if not form.errors:
                imei = request.args['imei']
                serial_no = request.args['serial_no']
                wifi_mac = request.args['wifi_mac'].replace(':', '')
                bt_mac = request.args['bt_mac'].replace(':', '')
                phone_id = Phone.calculate_phone_id(imei, serial_no, wifi_mac, bt_mac)
        return self.render('calc.html', form=form, result=phone_id, button_text='Calculate')


class PhoneUploadForm(Form):
    phones_data = FileUploadField(validators=[DataRequired()], description="File with phones list")
    batch_id = StringField(validators=[DataRequired()], description="Some ID for batch")
    allowed_gifts = SelectMultipleField(description="Allowed gifts", choices=zip(ALL_GIFTS, ALL_GIFTS),
                                        default=(PLUS_ID, DISK_ID))


class PhoneUploadView(AuthAdminView):
    DF_COLUMNS = ('IMEI_CODE', 'SECOND_IMEI_CODE', 'SERIAL_NUMBER',
                  'HASHED_VALUE', 'BLUETOOTH_CODE', 'WIFI_CODE')

    @expose('/', methods=('GET', 'POST'))
    def index(self, *args, **kwargs):
        form = PhoneUploadForm()
        if form.validate_on_submit():
            self.process_phones_data(
                data=form.phones_data.data,
                batch_id=form.batch_id.data,
                allowed_gifts=form.allowed_gifts.data,
            )
        return self.render('upload.html', form=form)

    def process_phones_data(self, data, batch_id, allowed_gifts):
        try:
            df = pd.read_csv(data, sep=';', dtype=str)
        except ValueError:
            flash('Invalid file format: should be the csv file with ";" separator and '
                  'following columns: %s' % ';'.join(self.DF_COLUMNS), category='error')
            return

        for column in self.DF_COLUMNS:
            if column not in df:
                flash('Column %s must be in CSV file' % column, category='error')
                return

        phones = []
        for _, row in df.iterrows():
            phone = Phone(
                imei=row['IMEI_CODE'],
                imei2=row['SECOND_IMEI_CODE'],
                serial_number=row['SERIAL_NUMBER'],
                id=row['HASHED_VALUE'],
                bluetooth_mac=row['BLUETOOTH_CODE'],
                wifi_mac=row['WIFI_CODE'],
                taxi_promocode=row.get('TAXI_PROMOCODE'),
                allowed_gifts=allowed_gifts,
                batch=batch_id,
            )
            try:
                phone.validate()
            except db.ValidationError as ex:
                flash('Phone check failed: %s' % format_error(ex), category='error')
                continue
            phones.append(phone)

        phone_ids = [p.pk for p in phones]

        existing_phone_ids = {p.pk for p in Phone.objects(pk__in=phone_ids)}
        for pk in existing_phone_ids:
            flash('Phone %s already in db' % pk, category='warning')

        phones = [p for p in phones if p.pk not in existing_phone_ids]

        if not phones:
            flash('No phones to upload', category='warning')
            return

        for phone in phones:
            logger.info('Inserting phone: %s', phone.to_json())
        Phone.objects.insert(phones)
        flash('Successfully uploaded %d phones to db' % len(phones))


def validate_buttons(form, field):
    primary_count = [b['style'] for b in field.data].count('primary')
    if primary_count == 0:
        raise ValidationError('No primary style button!')
    if primary_count > 1:
        raise ValidationError('More than one primary style button!')


def validate_single_button(form, field):
    if not field.data and form.data['style'] == 'primary':
        #  stop validation to prevent wtforms.validators.Optional from triggering (disables previous errors)
        raise StopValidation('This field is required for primary style')
    elif field.data and form.data['style'] != 'primary':
        raise ValidationError('This field must be empty for non-primary style')


class SettingsPromoBlockView(ModelView):
    access_roles = ROLE_DEVELOPER, ROLE_ADMIN, ROLE_RESOURCE
    loader = TranslationsAjaxLoader('title')

    form_subdocuments = {
        'image': get_s3_image_subdocument_config('settings_promo_block'),
        'buttons': {
            'form_subdocuments': {
                None: {
                    'form_ajax_refs': {
                        'caption': loader,
                    },
                    'form_extra_fields': {
                        'caption': AjaxSelectField(loader),
                    },
                    'form_args': {
                        'action': {
                            'validators': [validate_single_button],
                        },
                    },
                },
            },
        }
    }

    form_args = {
        'buttons': {
            'validators': [
                validate_buttons,
            ],
        },
        'experiment': {
            'label': 'Value',
            'description': 'Experiment promo_settings_block_experiment',
        },
    }

    column_formatters = {
        'image': format_s3_image,
    }

    column_list = ('id', 'experiment', 'priority', 'title', 'image', 'description', 'buttons', 'show_conditions')

    column_labels = {
        'experiment': 'Experiment value',
    }

    column_descriptions = {
        'experiment': 'Experiment promo_settings_block_experiment',
    }

    form_edit_rules = ('experiment', 'priority', 'title', 'image', 'description', 'buttons', 'show_conditions')

    form_widget_args = {
        'buttons': {
            'min_entries': 1,
            'max_entries': 2,
        },
        'priority': {
            'default': 0,
        },
    }

    form_ajax_refs = {
        'title': loader,
        'description': loader,
    }
    form_extra_fields = {
        'title': AjaxSelectField(loader),
        'description': AjaxSelectField(loader),
    }

    @staticmethod
    def _check_locale(key):
        collection = localization_mongo.db[app.config['LAUNCHER_TRANSLATIONS_COLLECTION']]
        values = collection.find_one({'_id': key})['values']
        available_languages = [v['conditions']['locale']['language'] for v in values]

        return 'en' in available_languages

    def validate_form(self, form):
        keys_to_translate = [form.data.get('title'), form.data.get('description')]
        keys_to_translate.extend([button.get('caption') for button in form.data.get('buttons', [])])
        for key in keys_to_translate:
            if key and not self._check_locale(key):
                flash('Key "%s" has no "EN" locale' % key)
                return None

        return super(SettingsPromoBlockView, self).validate_form(form)


class ArrangerExperimentConfigModelView(ModelView):
    model_form_converter = CustomModelConverter
    form_args = {'priority_groups': {'validators': [ListUniqueValidator()]}}
