from uuid import UUID

import numpy as np
from flask import request, flash
from flask_admin import expose
from flask_wtf import FlaskForm
from wtforms import SelectField, StringField, IntegerField, SelectMultipleField
from wtforms.validators import DataRequired, NumberRange

import jafar
from jafar.admin.models.base import ROLE_ADMIN, ROLE_DEVELOPER
from jafar.admin.validators import UUIDValidator
from jafar.admin.views import AuthAdminView
from jafar.pipelines.ids import FRAME_KEY_TARGET, FRAME_KEY_ADVISOR_MONGO_INSTALLS, FRAME_KEY_PREDICTIONS
from jafar.pipelines.pipeline import PipelineConfig, RecommenderNotReady
from jafar.pipelines.predefined import all_predefined_pipelines, predefined_arranger_pipelines
from jafar.utils.launcher import get_categories_for_apps
from jafar.utils.structarrays import DataFrame
from jafar.vanga.stats_getter import StatsContext, PersonalNoFallbackGroup
from jafar_yt.profile_dataframe_converter import UserProfileConverter

USER_CATEGORIES_KEY = 'USER_CATEGORIES'


def get_all_categories():
    return [cat for row in jafar.advisor_mongo.db.categories.find() for cat in row[u'google_name']]


def get_user_installs(device_id):
    profile = jafar.advisor_mongo.db.profile.find_one({'_id': device_id})
    if profile:
        try:
            return UserProfileConverter(profile).get_installs(False)['item']
        except UserProfileConverter.IncompleteProfile:
            pass
    return []


def get_user_categories(device_id):
    items = get_user_installs(device_id)
    return list(set(get_categories_for_apps(list(items)).values()))


class SideBySideForm(FlaskForm):
    user = StringField(validators=[UUIDValidator(), DataRequired()])
    recommenders = SelectMultipleField(choices=[(i, i) for i in ['empty', 'personal'] +
                                                sorted(all_predefined_pipelines.keys())])
    split_by = SelectField('Split by',
                           choices=[('category', 'category'), ('similar_to', 'similar_to'), ('n', 'empty')],
                           default='category')
    top_n = IntegerField(default=20, validators=[NumberRange(min=0)],
                         description="Prediction limit. If categories requested limits the group length.")
    requested_categories = SelectMultipleField(description="%s retrieves categories of user. "
                                                           "Use ctrl/cmd for multiselection." % USER_CATEGORIES_KEY)

    def __init__(self, *args, **kwargs):
        super(SideBySideForm, self).__init__(*args, **kwargs)
        self.requested_categories.choices = ([('None', 'None'), (USER_CATEGORIES_KEY, USER_CATEGORIES_KEY)] +
                                             [(_, _) for _ in sorted(get_all_categories())])
        self.requested_categories.data = [USER_CATEGORIES_KEY]

    class Meta:
        csrf = False


class SideBySideView(AuthAdminView):
    access_roles = ROLE_ADMIN, ROLE_DEVELOPER

    def get_predictions(self, pipeline_name, user, top_n, requested_categories, split_by):
        # normalize user DeviceID representation
        user = UUID(user)
        config = PipelineConfig(
            recommendation_mode='generate',
            online=True,
        )
        if pipeline_name == 'empty':
            return [], []
        if pipeline_name in predefined_arranger_pipelines:
            return self.get_arranger_predictions(pipeline_name=pipeline_name, user=user)
        if pipeline_name == 'personal':
            return self.get_personal_counts_prediction(user)
        pipe_creator = all_predefined_pipelines.get(pipeline_name)
        pipe = pipe_creator(pipeline_config=config,
                            storage=jafar.storage_wrapper.storage, top_n=top_n)
        categories = requested_categories[:]
        if 'None' in categories:
            categories = None
        elif USER_CATEGORIES_KEY in categories:
            categories.remove(USER_CATEGORIES_KEY)
            categories = list(set(get_user_categories(user) + categories))
        context = pipe.create_initial_context(country='RU',
                                              frames={FRAME_KEY_TARGET: DataFrame.from_dict({'user': [str(user)]})},
                                              requested_categories=categories)
        try:
            context = pipe.apply_blocks(False, context)
            predictions = context.data[FRAME_KEY_PREDICTIONS]
            basket = context.data[FRAME_KEY_ADVISOR_MONGO_INSTALLS]['item']
            idx = np.argsort(predictions['value'])[::-1]
            predictions = predictions[idx]
            if split_by in predictions.columns:
                return [(category, [(row['item'], row['value']) for row in frame])
                        for category, frame in predictions.groupby(split_by)], basket
            return [(None, [(row['item'], row['value']) for row in predictions])], basket

        except RecommenderNotReady:
            flash('Pipeline "%s" is not ready' % pipeline_name, 'error')
            return [], []

    @staticmethod
    def get_user_classnames(user_id):
        installs = get_user_installs(user_id)
        top_map = jafar.top_class_name_loader.load(installs, country='RU')
        return [u'{package_name}/{class_name}'.format(**app[0]) for app in top_map.itervalues()]

    def get_arranger_predictions(self, pipeline_name, user):
        # normalize user DeviceID representation
        config = PipelineConfig(
            recommendation_mode='generate',
            online=True,
        )
        if pipeline_name == 'empty':
            return [], []
        pipe_creator = all_predefined_pipelines.get(pipeline_name)
        pipe = pipe_creator(pipeline_config=config,
                            storage=jafar.storage_wrapper.storage)
        items = self.get_user_classnames(user)
        context = pipe.create_initial_context(country='RU',
                                              frames={FRAME_KEY_TARGET: DataFrame.from_dict({
                                                  'user': [str(user)] * len(items),
                                                  'item': items
                                              })})
        try:
            predictions = pipe.predict_top_n(context)
            predictions['item'] = [i.split('/')[0] for i in predictions['item']]
            idx = np.argsort(predictions['value'])[::-1]
            return [(None, predictions[idx][['item', 'value']])], predictions['item']
        except RecommenderNotReady:
            flash('Pipeline "%s" is not ready' % pipeline_name, 'error')
            return [], []

    @staticmethod
    def get_personal_counts_prediction(user_id):
        items = get_user_installs(user_id)
        res = PersonalNoFallbackGroup(StatsContext(user_id, [], items, {}, False)).get()
        res = [(i[0][0], i[1]['personal']) for i in res.iteritems()]
        res = sorted(res, key=lambda x: -x[1])
        return [(None, res)], items

    @expose('/')
    def index(self, *args, **kwargs):
        predictions = []
        basket = []
        split_by = None
        form = SideBySideForm(request.args)
        if request.args:
            form.validate()
            if not form.errors:
                user = request.args['user']
                split_by = request.args['split_by']
                requested_categories = request.args.getlist('requested_categories')
                for recommender in request.args.getlist('recommenders'):
                    top_n = int(request.args['top_n'])
                    try:
                        cur_predictions, cur_basket = self.get_predictions(recommender, user, top_n,
                                                                           requested_categories, split_by)
                        basket = basket or cur_basket
                        predictions.append((cur_predictions, recommender))
                    except Exception as e:
                        flash('Error occured with pipeline {}: {}'.format(recommender, e))
        return self.render('side_by_side.html',
                           predictions=predictions,
                           form=form,
                           split_by=split_by,
                           basket=basket)
