# -*- coding: utf-8 -*-
from uuid import UUID

from flask import Response, flash
from flask_admin import expose
from flask_wtf import FlaskForm
from mongoengine import Q
from wtforms import SelectField, StringField, SubmitField
from wtforms.validators import InputRequired, StopValidation

from jafar.admin.models.base import ROLE_ADMIN, ROLE_DEVELOPER, ROLE_SUPPORT, ROLE_ASSESSOR
from jafar.admin.views import AuthAdminView
from jafar.models.yandexphone import GiftSet, Phone, GreetingMail, DISK_ID
from jafar.utils import is_hex
from jafar.utils.passport import user_info_by_login, user_info_by_uid


class SupportViewValidator(object):

    def __call__(self, form, field):
        value = field.data
        value_type = form._fields['type'].data

        if value_type == 'device_id':
            try:
                UUID(value)
            except ValueError:
                raise StopValidation('Enter correct device_id (32 hex symbols)')

        elif value_type == 'phone_id':
            # phone_id is sha256
            if not is_hex(value) or len(value) != 64:
                raise StopValidation('Enter correct phone_id (64 hex symbols)')

        elif value_type == 'imei':
            if not value.isdigit() or len(value) != 15:
                raise StopValidation('Enter correct imei (15 digits)')

        elif value_type == 'uid':
            if not value.isdigit():
                raise StopValidation('Enter correct uid (digits)')

        elif value_type == 'serial_number':
            if not value.startswith('Y') or len(value) != 12:
                raise StopValidation('Enter correct serial number (12 symbols) starting with "Y"')


class SupportForm(FlaskForm):
    type = SelectField(u'Искать по', choices=[
        ('login', u'Логин или email'),
        ('device_id', u'Device ID'),
        ('uid', u'Паспортный uid'),
        ('imei', u'IMEI'),
        ('phone_id', u'Phone ID'),
        ('taxi_promocode', u'Промокод Такси'),
        ('serial_number', u'Серийный номер'),
    ])
    value = StringField(validators=[InputRequired(), SupportViewValidator()])
    submit = SubmitField(u'Искать')


class SupportView(AuthAdminView):
    access_roles = ROLE_ADMIN, ROLE_DEVELOPER, ROLE_SUPPORT, ROLE_ASSESSOR

    @expose('/', methods=('GET', 'POST'))
    def index(self):
        form = SupportForm()
        if form.validate_on_submit():
            search_type = form.data['type']
            value = form.data['value'].strip()

            data = {
                'user_info': None,
                'phone_info': None,
                'gifts_info': None
            }

            if search_type in ('login', 'uid'):
                # Searching user, then searching info about phone and gifts by uid
                user_info = find_user_info(search_type, value)
                if user_info:
                    data['user_info'] = user_info
                    uid = int(user_info['uid'])

                    gifts_info = find_gifts_info(uid=uid)
                    if gifts_info:
                        data['gifts_info'] = gifts_info

                    phone_info = find_phone_info('uid', uid)
                    if phone_info:
                        data['phone_info'] = phone_info
            elif search_type in ('device_id', 'phone_id', 'imei', 'taxi_promocode', 'serial_number'):
                # Searching info about phone, then info about gifts and user that got this gifts
                phone_info = find_phone_info(search_type, value)
                if phone_info:
                    data['phone_info'] = phone_info
                    uid = phone_info['gift_set']
                    if uid:
                        uid = int(uid)

                        user_info = user_info_by_uid(uid)
                        if user_info:
                            data['user_info'] = user_info

                        gifts_info = find_gifts_info(uid=uid)
                        if gifts_info:
                            data['gifts_info'] = gifts_info

            return self.render('support.html', form=form, **data)
        return self.render('support.html', form=form)

    @expose('/activate_taxi_promocode/<passport_uid>', methods=('POST',))
    def activate_taxi_promocode(self, passport_uid):
        gift_set = GiftSet.objects.get(passport_uid=passport_uid)
        gift_set.activate_gift('taxi')
        gift_set.save()
        return Response()


def find_user_info(search_type, value):
    user_info = {}
    if search_type == 'login':
        user_info = user_info_by_login(value)
    elif search_type == 'uid':
        user_info = user_info_by_uid(value)

    return user_info


gifts_labels = {
    'taxi': u'Такси',
    'money': u'Деньги',
    'plus': u'Плюс',
}

plus_subscribe_type_label = {
    'plus_auto_subscription': u'Автоматическая подписка в SUW',
    'plus_manual_subscription': u'Подписка через приложение Музыки',
    'plus_promocode': u'Подписка через промокод',

    'plus_only_plus_auto_subscription': u'Автоматическая подписка в SUW (только Плюс)',
    'plus_only_plus_manual_subscription': u'Подписка через приложение Музыки (только Плюс)',
    'plus_only_plus_promocode': u'Подписка через промокод (только Плюс)',
}


def find_gifts_info(uid):
    gift_set = GiftSet.objects.filter(passport_uid=uid).first()
    greeting_mail = GreetingMail.objects.filter(passport_uid=uid).first()

    gifts_info = {}
    if gift_set:
        gifts_info['passport_uid'] = gift_set.passport_uid
        gifts_info['created_at'] = gift_set.created_at
        gifts_info['gifts'] = {gifts_labels[item['gift'].pk]: item.to_mongo().to_dict() for item in gift_set.gifts}
        if not greeting_mail:
            flash(u'Ошибка: пользователь не получил письмо с подарками!')

    if greeting_mail:
        if 'no_gifts' in greeting_mail.mail_type:
            gifts_info['gifts_already_given'] = True
        else:
            gifts_info['plus_subscribe_type'] = plus_subscribe_type_label[greeting_mail.mail_type]
    return gifts_info


def find_phone_info(search_type, value):
    phone = None
    if search_type == 'phone_id':
        phone = Phone.objects.filter(id=value).first()
    elif search_type == 'imei':
        phone = Phone.objects.filter(Q(imei=value) | Q(imei2=value)).first()
    elif search_type == 'uid':
        phone = Phone.objects.filter(gift_set=value).first()
    elif search_type == 'taxi_promocode':
        phone = Phone.objects.filter(taxi_promocode=value).first()
    elif search_type == 'device_id':
        phone = Phone.objects.filter(device_id=value).first()
    elif search_type == 'serial_number':
        phone = Phone.objects.filter(serial_number=value).first()

    phone_info = {}
    import datetime
    if phone:

        allowed_gifts = phone.allowed_gifts
        if DISK_ID in allowed_gifts:
            allowed_gifts.remove(DISK_ID)

        phone_info = {
            'phone_id': phone.id,
            'activated': phone.activated,
            'first_activation': phone.first_activation_datetime or datetime.datetime.now(),
            'last_activation': phone.last_activation_datetime or datetime.datetime.now(),
            'gift_set': phone.gift_set.pk if phone.gift_set else None,
            'disk_autoupload': phone.disk_autoupload,
            'batch': phone.batch,
            'device_id': phone.device_id.hex if phone.device_id else None,
            'allowed_gifts': sorted([gifts_labels[gift_id] for gift_id in allowed_gifts]),
        }
    return phone_info
