# -*- coding: utf-8 -*-
import logging
from datetime import datetime

import requests
from flask import Markup, flash, redirect, url_for
from flask_admin.actions import action
from flask_admin.babel import gettext
from flask_admin.base import expose
from flask_admin.form import rules
from wtforms.fields import StringField

from jafar.admin.models.base import ROLE_ADMIN, ROLE_DEVELOPER, ROLE_RESOURCE
from jafar.admin.tanker import TANKER_SHOW_URL, TANKER_FAVICON, TranslationsUpdater
from jafar.admin.views.base import ModelView, IconLinkRowAction
from jafar.utils import fields_extractor
from jafar.utils.log import redirect_logging

logger = logging.getLogger('stream')

TANKER_KEYSETS_URL = 'https://tanker-api.yandex-team.ru/admin/project/{project}/keysets/?branch-name=master'


def download_keyset_list(project):
    response = requests.get(url=TANKER_KEYSETS_URL.format(project=project), timeout=5)
    response.raise_for_status()
    data = response.json()
    all_keysets = data['data']['items']
    backend_keysets = filter(lambda o: o['name'].startswith('backend_'), all_keysets)
    return map(lambda x: (project, x['name']), backend_keysets)


def formatter_datetime(view, context, model, name):
    value = model[name]
    return Markup(value.strftime('%Y-%m-%d<br>%H-%M-%S')) if value else ''


class TranslationsView(ModelView):
    access_roles = ROLE_ADMIN, ROLE_DEVELOPER, ROLE_RESOURCE
    can_create = True
    can_edit = True
    can_delete = True

    warnings = ''

    list_template = 'translations_list.html'

    form_rules = (
        rules.Field('title'),
        rules.Field('project'),
        rules.Field('keyset_id'),
        rules.Field('statuses'),
        rules.Field('description'),
    )

    datetime_fields = ('updated_at', 'created_at')

    column_formatters = {name: formatter_datetime for name in datetime_fields}

    column_extra_row_actions = [
        IconLinkRowAction(TANKER_FAVICON,
                          lambda view, row_id, row: TANKER_SHOW_URL.format(**row.to_mongo()),
                          u'Open translations in the Yandex.Tanker'),
    ]

    form_widget_args = {
        name: {'readonly': True} for name in datetime_fields
    }

    form_extra_fields = {name: StringField() for name in datetime_fields}

    @action('update', 'Update translations', 'Are you sure you want to update selected translations?')
    def action_update_translations(self, ids):
        try:
            selected = self.model.objects(id__in=ids)
            extractor = fields_extractor('project', 'keyset_id', 'statuses')
            translation_ids = [extractor(doc) for doc in selected]

            with redirect_logging('stream') as stream:
                self.update_translations(translation_ids)
                self.warnings = stream.getvalue().decode('utf-8')

            for doc in selected:
                doc.updated_at = datetime.utcnow()
                doc.save()
            flash(gettext('Translations was successfully updated.'), 'success')
        except Exception as ex:
            if not self.handle_view_exception(ex):
                raise
            flash(gettext('Failed to update translations. %(error)s', error=str(ex)), 'error')

    @staticmethod
    def update_translations(translation_ids):
        updater = TranslationsUpdater()
        updater.update(translation_ids)
        updater.commit()

    def clear_warnings(self):
        self.warnings = ''

    @expose('/import/', methods=('GET',))
    def import_view(self):
        with redirect_logging('stream') as stream:
            logger.info('keysets importing from Yandex.Tanker')
            model = self.model
            projects = set(model.objects.distinct('project'))
            projects.add('launcher')
            keysets = []
            for project in projects:
                keysets.extend(download_keyset_list(project))

            keysets_in_db = frozenset((doc['project'], doc['keyset_id']) for doc in model.objects.all())

            imported_count = 0
            for keyset in keysets:
                if keyset not in keysets_in_db:
                    project, keyset_id = keyset
                    title = keyset_id[len('backend_'):].replace('_', ' ').capitalize()
                    doc = self.model(
                        title=title,
                        project=project,
                        keyset_id=keyset_id,
                        description='keyset imported from Yandex.Tanker')
                    doc.save()
                    imported_count += 1
                    logger.info('imported %s', doc)

            if imported_count:
                flash('%d keyset(s) imported from Yandex.Tanker' % imported_count, category='success')
            else:
                flash('Keysets the same as in Yandex.Tanker', category='success')
            self.warnings = stream.getvalue().decode('utf-8')
            return redirect(url_for('.index_view'))
