from uuid import UUID

import httpgeobase
from flask import request, flash
from flask_admin import expose
from flask_wtf import Form
from wtforms import StringField
from wtforms.validators import DataRequired

from jafar import advisor_mongo
from jafar.admin.models.base import ROLE_ADMIN, ROLE_DEVELOPER, ROLE_TESTER, ROLE_SUPPORT
from jafar.admin.validators import UUIDValidator
from jafar.admin.views import AuthAdminView

geobase = httpgeobase.Lookup()


class FindUserForm(Form):
    user_id = StringField(validators=[DataRequired(), UUIDValidator()], description="User's DeviceID or UUID")

    class Meta:
        csrf = False


class UserInfoView(AuthAdminView):
    access_roles = ROLE_ADMIN, ROLE_DEVELOPER, ROLE_TESTER, ROLE_SUPPORT

    @staticmethod
    def get_region_name(region_id):
        try:
            return geobase.region_by_id(region_id).name.decode('utf8')
        except RuntimeError:
            return u"Unknown"

    @expose('/')
    def index(self, *args, **kwargs):
        form = FindUserForm(request.args)
        profile = None
        clients = None
        phone = None
        if request.args:
            form.validate()
            if not form.errors:
                user_id = request.args['user_id']
                profile = self.get_profile(user_id)
                if profile:
                    clients = self.get_clients(profile)
                    phone = self.get_phone(profile)
        return self.render('user_info.html', form=form, profile=profile, clients=clients, phone=phone)

    @staticmethod
    def get_clients(profile):
        return advisor_mongo.db['client'].find({'device_id': profile['_id']}, sort=[('updated_at', -1)])

    @staticmethod
    def get_profile(user_id):
        profile = advisor_mongo.db['profile'].find_one({'_id': UUID(user_id)})
        if profile is not None:
            return profile
        client = advisor_mongo.db['client'].find_one({'_id': UUID(user_id)})
        if client is not None and 'device_id' in client:
            profile = advisor_mongo.db['profile'].find_one({'_id': client['device_id']})
            if profile is not None:
                return profile
        flash('Profile is not found', category='warning')

    @staticmethod
    def get_phone(profile):
        phone = advisor_mongo.db['phone'].find_one({'device_id': profile['_id']})
        if phone:
            for k in phone.keys():
                if 'promocode' in k:
                    del phone[k]
            return phone
