import json
from collections import OrderedDict
from datetime import datetime
from io import BytesIO
from tempfile import NamedTemporaryFile
from uuid import UUID

import numpy as np
import pandas as pd
from bson import Binary
from catboost import CatBoostClassifier
from flask import request, Markup, send_file, flash, redirect
from flask_admin import expose
from flask_admin.actions import action
from flask_admin.form import DateTimePickerWidget, FileUploadInput
from flask_admin.form.upload import FileUploadField
from flask_wtf import FlaskForm
from wtforms import StringField, SelectField
from wtforms.validators import DataRequired
from wtforms.widgets import HTMLString, html_params

from jafar import fast_cache, jafar_mongo
from jafar.admin.fields import BaseLocalDateTimeField, MdsFileUploadField
from jafar.admin.models.base import ROLE_ADMIN, ROLE_DEVELOPER
from jafar.admin.validators import UUIDValidator
from jafar.admin.views import ModelView, AuthAdminView
from jafar.mongo_configs import VangaModel
from jafar.vanga.view import VangaView
from jafar_yt.utils.structarrays import DataFrame


class VangaForm(FlaskForm):
    user = StringField(validators=[UUIDValidator(), DataRequired()])
    dt = BaseLocalDateTimeField(label='DateTime',
                                default=datetime.utcnow(),
                                widget=DateTimePickerWidget(),
                                validators=[DataRequired()])
    model = SelectField()

    def __init__(self, *args, **kwargs):
        super(VangaForm, self).__init__(*args, **kwargs)
        self.model.choices = [(id_, doc.get('created_at')) for id_, doc in get_models().iteritems()]

    class Meta:
        csrf = False


class VangaDemoView(AuthAdminView):
    access_roles = ROLE_ADMIN, ROLE_DEVELOPER

    @expose('/')
    def index(self, *args, **kwargs):
        form = VangaForm(request.args)
        suggestions = []
        if request.args:
            form.validate()
            if not form.errors:
                user = UUID(form.data['user'])
                stats = VangaView.get_stats_from_mongo(user)
                suggestions = prepare_suggestions(stats, form.data['dt'], form.data['model'])

        return self.render('vanga.html', form=form, suggestions=suggestions)


def prepare_data(stats, dt):
    day = str(dt.weekday())
    hour = str(dt.hour)
    vectors = []

    for name, stat in stats.iteritems():
        vectors.append((stat['hourly'].get(hour, 0), stat['personal'], stat['weekly'].get(day, 0), -1))

    vectors = np.array(vectors)
    return (vectors - np.mean(vectors[:, 1]))/np.std(vectors[:, 1])


@fast_cache.memoize(300)
def get_models():
    return {str(doc['_id']): doc for doc in jafar_mongo.db.vanga_models.find()}


@fast_cache.memoize()
def get_catboost(_id):
    model = get_models()[_id]['content']
    with NamedTemporaryFile() as f:
        f.write(model)
        f.flush()
        return CatBoostClassifier().load_model(f.name)


def prepare_suggestions(stats, dt, model_id):
    if not stats:
        return []
    stats = OrderedDict(stats)
    data = prepare_data(stats, dt)

    model = get_catboost(model_id)
    scores = model.predict_proba(data)[:, 1]

    data = np.array(map(tuple, data), dtype=[(feature, np.float) for feature in ['hourly', 'personal', 'weekly', 'recent']])
    result = DataFrame.from_structarray(data)
    result = result.append_column(stats.keys(), 'package_name')
    result = result.append_column(scores, 'score')
    result = result.append_column([json.dumps(stat, indent=4, sort_keys=True) for stat in stats.itervalues()],
                                  'stats')
    return result[np.argsort(-scores)]


class BinaryInput(FileUploadInput):
    def __call__(self, field, **kwargs):
        kwargs.setdefault('id', field.id)
        kwargs.setdefault('name', field.name)

        return HTMLString(self.empty_template % {
            'file': html_params(type='file',
                                value='',
                                **kwargs),
        })


class BinaryUploadField(FileUploadField):
    widget = BinaryInput()

    def _save_file(self, data, filename):
        return Binary(data.read())


class VangaModelView(ModelView):
    form_overrides = dict(content=BinaryUploadField)
    form_excluded_columns = 'created_at',
    column_default_sort = 'created_at', True
    column_sortable_list = 'comment', 'created_at'

    def _dict_field_formatter(self, column, model, name):
        result = ''
        for key, value in sorted(model[name].items()):
            if value:
                result += '<tr><td>{0}</td><td align="left" style="padding-left: 10pt">{1:.3f}</td></tr>'.format(key, value)
        result = '<table class="table table-striped table-hover">{}</table>'.format(result)

        return Markup(result)

    def _download_formatter(self, context, model, name):
        return Markup(
            "<a href='{url}' target='_blank'>Download</a>".format(url=self.get_url('vangamodel.download', id=model.id)))

    def _workflow_formatter(self, context, model, name):
        return Markup("<a href={0} target='_blank'>Flow</a>".format(model.workflow))

    column_formatters = dict(
        workflow=_workflow_formatter,
        metrics=_dict_field_formatter,
        content=_download_formatter,
        features=_dict_field_formatter
    )

    @expose('/download_model/<id>')
    def download(self, id):
        model = VangaModel.objects.get(id=id)
        return send_file(
            BytesIO(model.content),
            attachment_filename='catboost.bin',
            mimetype='application/octet-stream',
            as_attachment=True
        )

    @action('compare', 'Compare models')
    def compare_models(self, ids):
        if len(ids) < 2:
            flash("Two or more models for comparison")
            return
        return redirect(self.get_url('vangamodel.compare', ids=','.join(ids)))

    @expose('/compare/<ids>')
    def compare(self, ids, *args, **kwargs):
        models = VangaModel.objects(id__in=ids.split(','))
        df = pd.DataFrame.from_records([m.metrics for m in models], index=[m.comment for m in models])
        df = df.transpose().fillna(0)
        result = df.style.set_precision(
            3
        ).set_table_styles([
            dict(selector="tr:hover", props=[("background-color", "#f2f2f2")]),
            dict(selector='td', props=[('padding', '10pt 10pt')])
        ]).set_table_attributes(
            'class="table table-striped table-hover"'
        ).highlight_max(
            color='#99ff99',
            axis=1
        )
        return self.render('comparison.html', table=Markup(result.render()))


def get_mds_vanga_model_filename(model, data):
    version = model.client_version
    return 'vanga-model-{version}'.format(version=version)


class ClassifierView(ModelView):
    column_list = ('vanga_model_name', )
    form_overrides = {'filename': MdsFileUploadField}
    form_args = {
        'filename': {
            'namegen': get_mds_vanga_model_filename,
            'kind': 'vanga_model'
        }
    }
