# -*- coding: utf-8 -*-
import logging

from flask import request, send_file, redirect, url_for, flash
from flask_admin.base import expose
from flask_admin.form.upload import FileUploadField
from flask_admin.model.fields import AjaxSelectField
from flask_admin.model.template import macro, EndpointLinkRowAction
from flask_wtf import FlaskForm
from wtforms.fields import StringField, BooleanField
from wtforms.validators import DataRequired

from jafar.admin.ajax import TranslationsAjaxLoader
from jafar.admin.fields import CompositeCoverField, ColorField
from jafar.admin.models.base import ROLE_ADMIN, ROLE_DEVELOPER, ROLE_RESOURCE
from jafar.admin.tanker import TANKER_FAVICON, TANKER_KEY_URL, TANKER_SHOW_URL
from jafar.admin.views.base import ModelView, IconLinkRowAction, get_s3_image_subdocument_config, get_md5_image_namegen

logger = logging.getLogger()

# advisor/launcher/models/wallpapers has the same template
CATEGORY_ID_TEMPLATE = 'backend_wallpapers_col_{category_name}'


def get_tanker_col_url(view, row_id, row):
    return TANKER_KEY_URL.format(
        keyset_id='backend_wallpapers',
        project='launcher',
        key_id=CATEGORY_ID_TEMPLATE.format(category_name=row.name)
    )


def get_tanker_show_url(*args):
    return TANKER_SHOW_URL.format(
        keyset_id='backend_wallpapers',
        project='launcher'
    )


class WallpaperImportForm(FlaskForm):
    file = FileUploadField(allowed_extensions=('zip',), validators=[DataRequired()])
    replace = BooleanField()


class WallpapersView(ModelView):
    access_roles = ROLE_DEVELOPER, ROLE_ADMIN, ROLE_RESOURCE

    can_copy = False
    page_size = 5
    can_set_page_size = True

    list_template = 'wallpapers_list.html'

    form_columns = ('name', 'covers', 'composite_cover', 'images', 'updated_at', 'use_in_autochange', 'use_in_feed')

    column_list = ('name', 'covers', 'composite_cover', 'images', 'use_in_autochange', 'use_in_feed')

    column_default_sort = [('name', True)]

    column_formatters = {
        'covers': macro('render_covers'),
        'images': macro('render_wallpapers'),
        'composite_cover': macro('render_generated_covers'),
    }

    column_extra_row_actions = [
        IconLinkRowAction(TANKER_FAVICON, get_tanker_col_url, u'Open translations for a category in Yandex.Tanker'),
        IconLinkRowAction(TANKER_FAVICON, get_tanker_show_url, u'Show all backend_wallpapers in Yandex.Tanker'),
        EndpointLinkRowAction('glyphicon glyphicon-download-alt', '.export_category')
    ]

    form_widget_args = {
        'updated_at': {'readonly': True}
    }

    cover_background_field = {
        'form_columns': ('image', 'color', 'has_gradient'),
        'form_extra_fields': {
            'color': ColorField()
        },
        'form_subdocuments': {
            'image': {
                'form_columns': ('key',),
                'form_extra_fields': {
                    'key': CompositeCoverField(
                        namegen=get_md5_image_namegen('wallpapers/cover_backgrounds'),
                        label='File'
                    ),
                }
            },
        },
    }

    form_extra_fields = {
        'name': StringField(),
        'updated_at': StringField()
    }

    # mongoengine ajax loader should have such formed title to be successfully exposed
    # The loader's title should represent a model's structure.
    loader = TranslationsAjaxLoader('composite_cover-title-text')

    translation_ajax_field = {
        'form_columns': ('text', 'color'),
        'form_ajax_refs': {
            'text': loader
        },
        'form_extra_fields': {
            'text': AjaxSelectField(loader, 'Key', allow_blank=True),
            'color': ColorField(),
        }
    }

    form_subdocuments = {
        'covers': {
            'form_subdocuments': {
                None: {
                    'form_subdocuments': {
                        'image_full': get_s3_image_subdocument_config('wallpapers/covers'),
                    },
                },
            },
        },
        'composite_cover': {
            'form_subdocuments': {
                'title': translation_ajax_field,
                'subtitle': translation_ajax_field,
                'background': cover_background_field,
            },
            'form_excluded_columns': ['padding_dp', 'generated_covers']
        },
        'images': {
            'form_subdocuments': {
                None: {
                    'form_subdocuments': {
                        'image': get_s3_image_subdocument_config('wallpapers/images')
                    },
                    'form_columns': ('priority', 'image', 'colors', 'badges', 'url_short'),
                    'form_widget_args': {
                        'url_short': {'readonly': True},
                    },
                },
            },
        },
    }

    @expose('/export/', methods=('GET',))
    def export_category(self):
        category_id = request.args['id']
        category = self.model.objects.get(id=category_id)
        zip_file = category.export()
        return send_file(
            zip_file,
            as_attachment=True,
            attachment_filename='%s.zip' % category.name,
        )

    @expose('/import/', methods=('GET', 'POST'))
    def import_category(self):
        form = WallpaperImportForm()
        if request.files:
            form.validate()
            self.model.import_(file_object=form.data['file'], replace=form.data['replace'])
            return redirect(url_for('.index_view'))
        return self.render('import.html', form=form)

    @expose('/reset/', methods=('POST',))
    def reset_view(self):
        flash('Wallpapers are restored from current API version', 'warning')
        self.model.reset()
        return redirect(url_for('.index_view'))

    @expose('/apply/', methods=('POST',))
    def apply_view(self):
        flash('Changes are delivered to API', 'success')
        self.model.apply()
        return redirect(url_for('.index_view'))


class WallpaperColorsView(ModelView):
    column_default_sort = 'order'


class MusicWallpapersView(ModelView):

    column_formatters = {
        'image': macro('render_s3_image'),
    }

    form_subdocuments = {
        'image': get_s3_image_subdocument_config('music_wallpapers'),
    }

    form_widget_args = {
        'updated_at': {
            'readonly': True,
        },
    }
