import logging

from flask import render_template
from werkzeug.datastructures import FileStorage
from wtforms.widgets import Input, HTMLString, html_params

import colorwiz.tool as colorwiz
from jafar import s3_client

logger = logging.getLogger(__name__)


def process_colorwiz_filestorage(data):
    value = data.filename
    color = colorwiz.process_image(value)['card_background']
    return value, color


def process_colorwiz_url(data):
    url = s3_client.get_url(data)
    logger.debug('get_image_and_color: %s', url)
    if s3_client.get_key(data):
        logger.debug('Trying to get color for a key: %s', data)
        color = colorwiz.process_url(url)['card_background']  # todo: timeout or something?
        return data, color
    else:
        logger.debug('No such key: %s', data)
        return '', ''


def get_image_and_color(data):
    if isinstance(data, basestring):
        return process_colorwiz_url(data)
    if isinstance(data, FileStorage):
        return process_colorwiz_filestorage(data)
    else:
        return '', ''


def make_params(**kwargs):
    return HTMLString(html_params(**kwargs))


class ColorWidget(Input):
    input_type = 'color'


class S3ImageUploadInput(object):
    """
        Renders a file input chooser field.

        You can customize `empty_template` and `data_template` members to customize
        look and feel.
    """
    empty_template = ('<input %(file)s>')

    data_template = ('<div>'
                     '<img %(image)s>'
                     '<input type="checkbox" name="%(marker)s">Delete</input>'
                     '</div>'
                     '<input %(file)s>')

    def __call__(self, field, **kwargs):
        kwargs.setdefault('id', field.id)
        kwargs.setdefault('name', field.name)

        template = self.data_template if field.data else self.empty_template

        if field.errors:
            template = self.empty_template

        if field.data and isinstance(field.data, FileStorage):
            value = field.data.filename
        else:
            value = field.data or ''

        return HTMLString(template % {
            'image': html_params(src=s3_client.get_url(value),
                                 width='200px'),
            'file': html_params(type='file',
                                value=value,
                                **kwargs),
            'marker': '_%s-delete' % field.name
        })


class CompositeCoverWidget(object):
    template = 'cover.html'

    def __call__(self, field, **kwargs):

        kwargs.setdefault('id', field.id)
        kwargs.setdefault('name', field.name)

        value, color = get_image_and_color(field.data)

        if not field.data or field.errors:
            return render_template(
                self.template,
                file=make_params(type='file', value=value, **kwargs),
                color=make_params(type='hidden', value=color),
                img_block=make_params(style="display: none"),
            )
        else:
            return render_template(
                self.template,
                image=make_params(id='image-input', src=s3_client.get_url(value), width='200px'),
                file=make_params(type='file', value=value, **kwargs),
                color=make_params(type='hidden', value=color),
                delete=make_params(type='checkbox', name='_%s-delete' % field.name),
            )
