import logging
from operator import itemgetter

import gevent
import yt.wrapper as yt
from OpenSSL import SSL
from flask import current_app as app
from werkzeug.exceptions import ServiceUnavailable
from yt.packages.requests.exceptions import RequestException, BaseHTTPError

logger = logging.getLogger(__name__)

DYNAMIC_TABLES_CONFIG = {
    'proxy': {
        'retries': {
            'count': 1,
        },
        'request_timeout': 500,
        'connect_timeout': 100,
    },
    'dynamic_table_retries': {
        'enable': False,
    },
    # 'backend': 'rpc',
}

# hard timeout for YT wrapper to instantiate YtClient and make a call
YT_REQUEST_TIMEOUT = 3

# Possible errors that can be thrown from the yt call
YT_ERRORS = yt.YtError, RequestException, BaseHTTPError, SSL.Error

FALLBACK_LANGUAGES = ['en', 'ru']


class LoaderException(ServiceUnavailable):
    description = "Loader backend is unavailable"


class AppInfoNotFound(LoaderException):
    description = "App Info not found"


class VangaStatsNotFound(LoaderException):
    description = "Vanga Stats not found"


class YtRequestGreenlet(gevent.Greenlet):
    def __init__(self, proxy, token, *args, **kwargs):
        self.proxy = proxy
        self.token = token
        super(YtRequestGreenlet, self).__init__(None, *args, **kwargs)

    def _run(self, command, *args, **kwargs):
        # We have to create YtClient every time to prevent deadlocks because YtClient is not thread-safe
        client = yt.YtClient(proxy=self.proxy, config=DYNAMIC_TABLES_CONFIG, token=self.token)
        method = getattr(client, command)
        return method(*args, **kwargs)


def call_multiple_yt_clusters(command, *args, **kwargs):
    """"
    Spawns greenlets with same request to different Yt proxies.
    Returns first successful result.
    """
    greenlets = [YtRequestGreenlet.spawn(proxy, app.config['YT_TOKEN'], command, *args, **kwargs)
                 for proxy in itemgetter('YT_PROXY', 'YT_RESERVE_PROXY')(app.config)]
    try:
        for greenlet in gevent.iwait(greenlets, timeout=YT_REQUEST_TIMEOUT):
            # noinspection PyBroadException
            try:
                return greenlet.get()
            except YT_ERRORS as e:
                logger.warning('YT error on %s: %s', greenlet.proxy, e)
            except Exception:
                logger.exception('Unhandled exception')
        else:
            logger.error('YT request to all proxies is failed')
            raise LoaderException
    finally:
        # Kill all greenlets that is not finished yet, but not needed anymore
        gevent.killall(greenlets, block=False)


class Loader(object):
    def init_app(self, app):
        from jafar import cache
        self.cache = cache

    def load_async(self, **kwargs):
        return call_multiple_yt_clusters('lookup_rows', **kwargs)

    @staticmethod
    def make_cache_key(obj):
        raise NotImplementedError

    def process_object(self, obj):
        pass

    def _load_from_yt(self, query):
        lookuper = self.load_async(
            table=self.table,
            input_stream=query,
            format=yt.format.YsonFormat('binary')
        )
        result = []
        for obj in lookuper:
            self.process_object(obj)
            self.cache.set(self.make_cache_key(obj), obj)
            yield obj

    def _load_from_cache(self, query):
        for obj in query:
            key = self.make_cache_key(obj)
            app_info = self.cache.get(key)
            if app_info:
                yield app_info


class AppInfoLoader(Loader):
    NotFound = AppInfoNotFound

    def init_app(self, app):
        super(AppInfoLoader, self).init_app(app)
        self.table = app.config['YT_PATH_DYNAMIC_LINK']

    def process_object(self, app_info):
        for k, v in app_info.items():
            if isinstance(v, str):
                app_info[k] = v.decode('utf8')

    @staticmethod
    def make_cache_key(obj):
        return "app_info_cache_{}_{}".format(obj['package_name'], obj['language'])

    def load(self, package_names, language=None, raise_on_missing=True):
        unknown_apps = set(package_names)
        app_info_map = dict()
        languages = [language] + FALLBACK_LANGUAGES if language else FALLBACK_LANGUAGES
        for lang in languages:
            for loader in (self._load_from_cache, self._load_from_yt):
                query = [{'package_name': app, 'language': lang} for app in unknown_apps]
                for info in loader(query):
                    app_info_map[info['package_name']] = info
                    unknown_apps.discard(info['package_name'])
                if not unknown_apps:
                    return app_info_map
        if raise_on_missing:
            for package_name in unknown_apps:
                logger.error('App Info %s for %s not found', package_name, language)
            raise self.NotFound
        return app_info_map


class VangaGeneralStatsLoader(Loader):
    NotFound = VangaStatsNotFound

    def init_app(self, app):
        super(VangaGeneralStatsLoader, self).init_app(app)
        self.table = app.config['VANGA_GENERAL_DYNAMIC_PATH']

    @staticmethod
    def make_cache_key(obj):
        return u'vanga_stats_cache_{}/{}'.format(obj['package_name'], obj.get('class_name', ''))

    def load_async(self, **kwargs):
        table = kwargs.pop('table')
        query = kwargs.pop('input_stream')

        packages = ['"%s"' % item['package_name'] for item in query]
        query = 'package_name in (%s)' % ','.join(packages)

        kwargs['query'] = '* FROM [{table}] WHERE {query}'.format(
            table=table,
            query=query
        )

        return call_multiple_yt_clusters('select_rows', **kwargs)

    def load(self, package_names, raise_on_missing=False):
        unknown_apps = set(package_names)
        app_info_dict = dict()

        for loader in (self._load_from_cache, self._load_from_yt):
            query = [{'package_name': app_} for app_ in unknown_apps]
            for info in loader(query):
                app_info_dict.setdefault(info['package_name'], []).append(info)
                unknown_apps.discard(info['package_name'])
            if not unknown_apps:
                return app_info_dict

        if raise_on_missing and unknown_apps:
            for package_name in unknown_apps:
                logger.error('Vanga general stats for %s not found', package_name)
            raise self.NotFound

        return app_info_dict


class TopClassLoader(VangaGeneralStatsLoader):
    @staticmethod
    def make_cache_key(obj):
        return 'top_class_name_{}'.format(obj['package_name'])

    def load(self, package_names, country, raise_on_missing=False):
        self.table = app.config['ARRANGER_TOP_CLASS_DYNAMIC'][country]
        return super(TopClassLoader, self).load(package_names, raise_on_missing)
