import logging
from itertools import izip, ifilter

import numpy as np

import jafar.mongo_configs.arranger as config

logger = logging.getLogger(__name__)


class PlaceholderFillInIterator(object):
    """
    Takes placeholder and iterates coordinates of items to fill in
    """

    def __init__(self, placeholder):
        self.placeholder = placeholder

    def __iter__(self):
        raise NotImplementedError


class FromLowerLeftOrder(PlaceholderFillInIterator):
    def __iter__(self):
        for y in xrange(self.placeholder['cell']['y'] + self.placeholder['span']['y'] - 1,
                        self.placeholder['cell']['y'] - 1,
                        -1):
            for x in xrange(self.placeholder['cell']['x'],
                            self.placeholder['cell']['x'] + self.placeholder['span']['x']):
                yield x, y


class FromLowerLeftSnakeOrder(PlaceholderFillInIterator):
    def __iter__(self):
        y_start = self.placeholder['cell']['y'] + self.placeholder['span']['y'] - 1
        for y in xrange(y_start, self.placeholder['cell']['y'] - 1, -1):
            if (y_start - y) % 2 == 0:  # every second row change direction
                x_iter = xrange(self.placeholder['cell']['x'],
                                self.placeholder['cell']['x'] + self.placeholder['span']['x'])
            else:
                x_iter = xrange(self.placeholder['cell']['x'] + self.placeholder['span']['x'] - 1,
                                self.placeholder['cell']['x'] - 1,
                                -1)
            for x in x_iter:
                yield x, y


class FromLowerLeftChooseOrder(PlaceholderFillInIterator):
    def __iter__(self):
        if self.placeholder['rearrange']:
            order_iterator = iter(FromLowerLeftSnakeOrder(self.placeholder))
        else:
            order_iterator = iter(FromLowerLeftOrder(self.placeholder))
        return order_iterator


class HomescreensArrangement(object):
    """Takes arrangement template from config and fills it with pipeline output"""
    order = FromLowerLeftChooseOrder

    def __init__(self, config, user):
        self.user = user
        self.height = user.grid_height
        self.width = user.grid_width
        self.mask = None
        self.screens = []  # zero screen is dock
        self.items_filter = set()
        self.defaults = {obj['category']: obj['package'] for obj in user.defaults}
        self.priority_groups = [group for group in config.get('priority_groups', [])]
        self.prepare_grid([config['dock_template']] + config['screens_template'])

    @staticmethod
    def mask_coordinates(item):
        return ((item['cell']['y'], item['cell']['y'] + item['span']['y']),
                (item['cell']['x'], item['cell']['x'] + item['span']['x']))

    def prepare_grid(self, template_config):
        """Normalize config to the specific arrangement, prepare mask for conflicts resolving"""
        self.mask = np.zeros((len(template_config), self.height, self.width), dtype=np.bool)
        for screen_number, screen_template in enumerate(template_config):
            current_screen_items = []
            for item in screen_template['items']:
                self._normalize_coordinates(item)
                (y, r_y), (x, r_x) = self.mask_coordinates(item)
                # skip item if it is out of valid places
                if x < 0 or y < 0 or r_y <= y or r_x <= x or r_x > self.width or r_y > self.height:
                    continue

                # 0,0 is top left corner
                # skip item if place is busy, placeholder can flow around busy places
                if item['item_type'] != config.PLACEHOLDER_TYPE and np.any(self.mask[screen_number, y:r_y, x:r_x]):
                    continue

                if item['item_type'] == config.DEFAULTS_TYPE:
                    if item['value'] in self.defaults:
                        item['value'] = self.defaults[item['value']]  # transform default_category into app type
                        item['item_type'] = config.APP_ITEM_TYPE
                    else:
                        item['item_type'] = config.PLACEHOLDER_TYPE

                # skip if user doesn't have that item (except widgets, they are not in query body)
                # or if this item was before
                if item['item_type'] == config.APP_ITEM_TYPE:
                    if (item['value'] in self.items_filter or
                            (item['value'] not in self.user.packages and
                             item['value'] not in self.defaults.values())):
                        continue  # don't add this item to screen
                    self.items_filter.add(item['value'])
                    self.mask[screen_number, y, x] = True

                if item['item_type'] == config.WIDGET_ITEM_TYPE:
                    if item['value'] in self.items_filter:
                        continue  # don't add this item to screen
                    self.items_filter.add(item['value'])
                    self.mask[screen_number, y:r_y, x:r_x] = True

                current_screen_items.append(item)
            self.screens.append(current_screen_items)

    def _normalize_coordinates(self, item):
        """Reduce item size if it extends the screen and invert negative coordinates"""
        if item['cell']['x'] < 0:
            item['cell']['x'] = self.width + item['cell']['x']
        if item['cell']['y'] < 0:
            item['cell']['y'] = self.height + item['cell']['y']
        # resize span
        if item['cell']['x'] + item['span']['x'] > self.width:
            item['span']['x'] = self.width - item['cell']['x']

        if item['cell']['y'] + item['span']['y'] > self.height:
            item['span']['y'] = self.height - item['cell']['y']

    def fill_in(self, rec_groups):
        """Iterates over rec groups and fills in empties in placeholders"""
        apps_iter = self._apply_priority(rec_groups)
        result = []

        for screen_number, screen in enumerate(self.screens):
            current_screen_items = []
            for item in screen:
                if item['item_type'] != config.PLACEHOLDER_TYPE:  # remain constants as is
                    current_screen_items.append(item)
                    continue
                order = ((x, y) for (x, y) in self.order(item) if not self.mask[screen_number, y, x])
                # dangerous place: izip pops one element from first iterator even when second iterator ended
                for (x, y), app in izip(order, apps_iter):
                    item = make_item(app, x, y)
                    current_screen_items.append(item)
                    self.mask[screen_number, y, x] = True
            result.append(current_screen_items)

        self.screens = result

    def _apply_priority(self, rec_groups):
        """remain only most important app in group and put them upper"""
        recommended = [item for group in rec_groups for item in group.item_iter]  # all apps
        indices = [len(recommended)] * len(self.priority_groups)  # indices in large array
        found_app = [None] * len(self.priority_groups)  # prior apps themselves

        for group_index, group in enumerate(self.priority_groups):
            for prior_app in group['apps']:
                for overall_index, rec_app in enumerate(recommended):
                    if not rec_app or rec_app.split('/')[0] != prior_app:  # skip if no app found
                        continue
                    indices[group_index] = min(overall_index, indices[group_index])  # memoize its position
                    found_app[group_index] = found_app[group_index] or rec_app  # memoize prior app
                    recommended[overall_index] = None  # mark this position to remove
                    break

        shift = 0
        for index, app, group in zip(indices, found_app, self.priority_groups):
            if app:  # is it in recommended
                if group['boost']:
                    recommended.insert(0, app)  # put it to the top
                    shift += 1
                else:
                    recommended[index + shift] = app  # put it to the best place of group members

        return ifilter(bool, recommended)

    def get_rearrangement_ranges(self):
        """returns list of ranges in final arrangement"""
        items_sizes = []
        rearrange = []
        for screen_number, screen in enumerate(self.screens):
            for item in screen:
                if item['item_type'] == config.PLACEHOLDER_TYPE:
                    (y, r_y), (x, r_x) = self.mask_coordinates(item)
                    items_sizes.append(np.sum(~self.mask[screen_number, y:r_y, x:r_x]))
                    rearrange.append(item['rearrange'])
        items_sizes = np.concatenate(([0], np.cumsum(items_sizes)))
        return np.c_[items_sizes[:-1], items_sizes[1:]][rearrange].tolist()


class ArrangerStrategy(object):
    def __init__(self, config, user):
        self.config = config
        self.user = user

    def get_arrangement(self):
        arrangement = HomescreensArrangement(config=self.config, user=self.user)
        rearrangement_ranges = arrangement.get_rearrangement_ranges()
        rec_config = self.config['recommender_config']
        recommender = rec_config.recommender(user=self.user,
                                             pipeline=rec_config.pipeline,
                                             rearrangement_ranges=rearrangement_ranges,
                                             used_apps=list(arrangement.items_filter))
        groups = recommender.get_recommendation_groups(None, None)

        arrangement.fill_in(groups)
        return arrangement


def make_item(item, x, y):
    """Create dict with all needed fields"""
    return config.ScreenItem(cell=config.Position(x=x, y=y), value=item).to_mongo().to_dict()
