import datetime
import time
import logging
from uuid import UUID
from collections import defaultdict

import yt.wrapper as yt
from flask import current_app as app
from flask_script import Command, Option
from flask_clickhouse import ClickhouseError

from jafar import advisor_replica_mongo, clickhouse
from jafar.utils import date_to_datetime
from jafar.utils.io import get_cluster
from jafar.utils.iter import take
from jafar_yt.profile_dataframe_converter import mongo_to_yt

logger = logging.getLogger(__name__)

DEFAULT_LOOKUP_DAYS = 2
BATCH_SIZE = 10000

CLICKHOUSE_COLUMNS = (  # (name, type, default_value)
    ('device_id', 'UUID', UUID(int=0)),
    ('_id', 'String', ''),
    ('batch', 'String', ''),
    ('first_activation_datetime', 'DateTime', datetime.datetime.fromtimestamp(0)),
    ('last_activation_datetime', 'DateTime', datetime.datetime.fromtimestamp(0)),
    ('activated', 'UInt8', 0)
)


def users_generator(start_date=None, end_date=None):
    logger.info("Fetching users from %s to %s", start_date, end_date)
    query = defaultdict(dict)
    if start_date:
        query['updated_at']['$gte'] = date_to_datetime(start_date)
    if end_date:
        query['updated_at']['$lte'] = date_to_datetime(end_date)
    for i, profile in enumerate(advisor_replica_mongo.db.profile.find(dict(query)), 1):
        doc = mongo_to_yt(profile)
        doc['device_id'] = doc['_id']['$$uuid']
        yield doc
        if i % 10000 == 0:
            logger.info("%s users fetched", i)


class UpdateUsersYt(Command):

    option_list = (
        Option('--start_date', dest='start_date', default=None, action='store'),
        Option('--end_date', dest='end_date', default=None, action='store'),
    )

    def get_date_range(self, start_date, end_date):
        if not start_date and not end_date:
            now = datetime.datetime.utcnow().date()
            return now - datetime.timedelta(days=DEFAULT_LOOKUP_DAYS), None
        if start_date:
            start_date = datetime.datetime.strptime(start_date, '%Y-%m-%d').date()
        if end_date:
            end_date = datetime.datetime.strptime(end_date, '%Y-%m-%d').date()
        return start_date, end_date

    def run(self, start_date, end_date):
        yt.update_config(app.config['YT_CONFIG'])
        users_path = app.config['YT_PATH_USERS_FULL']
        start_date, end_date = self.get_date_range(start_date, end_date)
        iterator = iter(users_generator(start_date, end_date))

        with yt.TempTable() as temp_path:
            temp_path = yt.TablePath(temp_path, append=True)
            logger.info("Writing data to %s", temp_path)
            while True:
                batch = take(iterator, BATCH_SIZE)
                if not batch:
                    break
                yt.write_table(
                    temp_path, batch,
                    format=yt.JsonFormat(attributes={"encode_utf8": False})
                )

            yt.run_merge([users_path, temp_path], users_path)
            yt.run_sort(users_path, sort_by=['device_id'])

        def user_attributes_reducer(groups):
            for result, records in groups:
                yield max(records, key=lambda x: x['updated_at']['$date'])

        cluster = get_cluster()
        job = cluster.job()
        job.table(
            users_path
        ).groupby(
            'device_id'
        ).reduce(
            user_attributes_reducer,
        ).put(
            users_path
        )
        job.run()


class UpdatePhonesYt(Command):
    @staticmethod
    def lines_generator():
        for i, profile in enumerate(advisor_replica_mongo.db.phone.find(), 1):
            yield mongo_to_yt(profile)
            if i % 10000 == 0:
                logger.info("%s phones fetched", i)

    def run(self):
        yt.update_config(app.config['YT_CONFIG'])
        path = app.config['YT_PATH_PHONES']
        logger.info("Writing data to %s", path)
        yt.write_table(
            path, self.lines_generator(),
            format=yt.JsonFormat(attributes={"encode_utf8": False})
        )

class UpdatePhonesClickHouse(Command):
    """
        Writes activated(!) phones device_ids to ClickHouse table
    """

    @staticmethod
    def create_new_table():
        timestamp = int(time.time() * 1000000)  # Current timestamp for unique path in zookeeper
        clickhouse.execute("""
            DROP TABLE IF EXISTS {db}.{table}_new
            ON CLUSTER {cluster}
            """.format(db=app.config['CLICKHOUSE_DATABASE'],
                       table=app.config['CLICKHOUSE_TABLE_PHONES'],
                       cluster=app.config['CLICKHOUSE_CLUSTER'])
        )
        clickhouse.execute("""
            CREATE TABLE {db}.{table}_new
            ON CLUSTER {cluster}
            (
                date Date,
                {columns}
            ) ENGINE = ReplicatedMergeTree('/clickhouse/tables/{table}_{ts}', '{{replica}}')
            PARTITION BY date
            ORDER BY (date, {sorting_key})
            """.format(columns=',\n'.join(column + ' ' + type for column, type, _ in CLICKHOUSE_COLUMNS),
                db=app.config['CLICKHOUSE_DATABASE'], table=app.config['CLICKHOUSE_TABLE_PHONES'],
                ts=timestamp, cluster=app.config['CLICKHOUSE_CLUSTER'], sorting_key=CLICKHOUSE_COLUMNS[0][0])
        )

    @staticmethod
    def drop_old():
        clickhouse.execute("""
                        DROP TABLE
                        {db}.{table}_old
                        ON CLUSTER {cluster}
                        """.format(db=app.config['CLICKHOUSE_DATABASE'],
                                   table=app.config['CLICKHOUSE_TABLE_PHONES'],
                                   cluster=app.config['CLICKHOUSE_CLUSTER'])
                           )

    def rename_and_delete(self):
        query = """SHOW TABLES FROM {db}""".format(db=app.config['CLICKHOUSE_DATABASE'])
        tables = [str(table[0]) for table in clickhouse.execute(query)]
        old_table_exists = app.config['CLICKHOUSE_TABLE_PHONES'] + '_old' in tables
        initial_table_exists = app.config['CLICKHOUSE_TABLE_PHONES'] in tables
        if old_table_exists:
            self.drop_old()
        if initial_table_exists:
            clickhouse.execute("""
                RENAME TABLE
                {db}.{table} to {db}.{table}_old,
                {db}.{table}_new to {db}.{table}
                ON CLUSTER {cluster}
                """.format(db=app.config['CLICKHOUSE_DATABASE'],
                           table=app.config['CLICKHOUSE_TABLE_PHONES'],
                           cluster=app.config['CLICKHOUSE_CLUSTER'])
            )
            self.drop_old()
        else:
            clickhouse.execute("""
                RENAME TABLE
                {db}.{table}_new to {db}.{table}
                ON CLUSTER {cluster}
                """.format(db=app.config['CLICKHOUSE_DATABASE'],
                           table=app.config['CLICKHOUSE_TABLE_PHONES'],
                           cluster=app.config['CLICKHOUSE_CLUSTER'])
                   )

    def run(self):
        self.create_new_table()

        today = datetime.date.today()
        data = []
        for profile in advisor_replica_mongo.db.phone.find(projection=zip(*CLICKHOUSE_COLUMNS)[0]):
            data.append(
                tuple([today] + [profile.get(key, default) for key, _, default in CLICKHOUSE_COLUMNS])
            )

        logger.info("Writing data to ClickHouse: %s", app.config['YT_PATH_PHONES'])
        clickhouse.execute("""
            INSERT INTO {db}.{table}_new
            VALUES {values}
            """.format(db=app.config['CLICKHOUSE_DATABASE'],
                       table=app.config['CLICKHOUSE_TABLE_PHONES'],
                       values=zip(*CLICKHOUSE_COLUMNS)[0]),
            data
        )

        self.rename_and_delete()
