import logging
from datetime import date

import yt.wrapper as yt
from flask import current_app as app
from flask_script import Command
from nile.api.v1 import filters as nf, extractors as ne
from qb2.api.v1 import filters as qf
from yt.wrapper import yson

from jafar.datasets import VangaDatasetProcessor
from jafar.utils.dyntables import create_dyntable, dynamize_static_and_cleanup
from jafar.utils.io import get_cluster
from jafar_yt.utils.helpers import day_before
from jafar_yt.vanga_utils import GeneralReducer

logger = logging.getLogger(__name__)


SCHEMA = yson.YsonList([
    {"name": "package_name", "type": "string", "sort_order": "ascending"},
    {"name": "class_name", "type": "string", "sort_order": "ascending"},
    {"name": "personal", "type": "int64"},
    {"name": "hourly", "type": "any"},
    {"name": "weekly", "type": "any"}
])
SCHEMA.attributes["unique_keys"] = True
MODE = 'uncompressed'


def prepare_dynamic_table():
    result_table = yt.ypath_join(app.config['VANGA_GENERAL_STATIC_DIR'], date.today().isoformat())
    min_users = app.config['VANGA_MIN_USERS_FOR_APP']

    if not yt.exists(result_table):
        create_dyntable(result_table, yt, schema=SCHEMA, in_memory_mode=MODE)

    job = get_cluster().job()

    VangaDatasetProcessor.prepare_app_launches(
        job,
        day_before(app.config['VANGA_GENERAL_DAYS']).isoformat()
    ).project(
        'user',
        'hour',
        'weekday',
        package_name='packageName',
        class_name='className'
    ).filter(
        qf.nonzero('class_name')
    ).groupby(
        'package_name',
        'class_name'
    ).reduce(
        GeneralReducer('package_name', 'class_name')
    ).filter(
        nf.custom(lambda x: x > min_users, 'users')
    ).project(
        ne.all('users')
    ).sort(
        'package_name',
        'class_name'
    ).put(result_table)

    job.run()
    logger.debug('Reducing chunks size')
    yt.run_merge(
        source_table=result_table,
        destination_table=result_table,
        spec=dict(
            force_transform=True,
            combine_chunks=True,
            job_io=dict(
                table_writer=dict(
                    block_size=256 * 2 ** 10,
                    desired_chunk_size=100 * 2 ** 20
                ))))

    return result_table


class UpdateVangaStats(Command):
    def run(self):
        yt.update_config(app.config['YT_CONFIG'])
        table = prepare_dynamic_table()
        kwargs_list = [dict(base_path=app.config['VANGA_GENERAL_STATIC_DIR'],
                            ttl=app.config['VANGA_GENERAL_TTL'])]
        dynamize_static_and_cleanup(table=table,
                                    schema=SCHEMA,
                                    mount=True,
                                    symlink=app.config['VANGA_GENERAL_DYNAMIC_PATH'],
                                    cleanup_kwargs_list=kwargs_list,
                                    in_memory_mode=MODE)
