from flask import current_app as app
from flask_script import Command, Option

from jafar import storage_wrapper
from jafar.utils import io

import datetime
import logging
import os
from subprocess import check_output
from sandbox_common import rest
from sandbox_common.proxy import OAuth
import time

logger = logging.getLogger(__name__)


class SkynetUploader(object):

    def upload(self, name):
        resource_id = check_output(["sky", "share", "-d", app.config['SNAPSHOT_FOLDER'], name]).strip()
        io.cleanup_files_local(app.config['SNAPSHOT_FOLDER'], app.config['SNAPSHOT_LIMIT'])
        return resource_id


class SandboxUploader(object):
    polling_interval = 10

    def __init__(self):
        self.client = rest.Client(auth=OAuth(app.config['SANDBOX_OAUTH_TOKEN']))

    def wait_for_task(self, task_id, timeout=60 * 60):
        start = time.time()
        while True:
            status = self.client.task[task_id].read()['status']
            if status in ("FAILURE", "NO_RES", "EXCEPTION", "TIMEOUT", "STOPPED", "DELETED"):
                raise Exception("Task failed with status {}".format(status))

            if status == 'SUCCESS':
                logging.info("Task succeeded")
                break

            logger.info("Checking task status: %s", status)
            if time.time() - start > timeout:
                raise Exception("Timeout waiting for task: last status was {}".format(status))
            time.sleep(self.polling_interval)

    def retrieve_resource(self, task_id):
        resources = self.client.task[task_id].resources.read()['items']

        return str([r['id'] for r in resources if r['type'] == 'OTHER_RESOURCE'][0])

    def upload(self, name, skynet_resource):
        task_id = self.client.task({
            "owner": "ADVISOR",
            "author": app.config['ROBOT_LAUNCHER_YT_LOGIN'],
            "priority": {
                "class": "SERVICE",
                "subclass": "NORMAL"
            },
            "type": "REMOTE_COPY_RESOURCE",
            "description": "Uploading jafar snapshot: {}".format(name)
        })['id']
        logger.info("Created task %s", task_id)
        self.client.task[task_id] = {
            "custom_fields": [
                {"name": "created_resource_name", "value": name},
                {"name": "remote_file_name", "value": skynet_resource},
                {"name": "store_forever", "value": True},
                {"name": "resource_type", "value": "OTHER_RESOURCE"}
            ]
        }
        self.client.batch.tasks.start.update([task_id])
        self.wait_for_task(task_id)
        return self.retrieve_resource(task_id)


class UploadSnapshot(Command):
    """
    Dumps storage to a file and uploads it to Skynet and afterwards to Sandbox.
    """

    option_list = (
        Option(
            '--output', dest='output',
            type=str, action='store', help='Optional filename to write result'
        ),
    )

    @staticmethod
    def get_snapshot_name():
        return 'jafar_{version}_{date}.tar.gz'.format(
            version=app.config['SNAPSHOT_VERSION'],
            date=datetime.datetime.utcnow().strftime('%Y-%m-%dT%H-%M-%S')
        )

    def run(self, output=None):
        if not os.path.exists(app.config['SNAPSHOT_FOLDER']):
            os.makedirs(app.config['SNAPSHOT_FOLDER'])

        name = self.get_snapshot_name()
        path = os.path.join(app.config['SNAPSHOT_FOLDER'], name)
        storage_wrapper.storage.dump(path)

        # upload to Skynet first
        logging.info('Uploading %s to skynet', path)
        skynet_resource = SkynetUploader().upload(name)
        logger.info(skynet_resource)

        # then from skynet to sandbox
        resource_id = SandboxUploader().upload(name, skynet_resource)

        logger.info("Snadbox resource with snapshot: %s", resource_id)

        if output:
            with open(output, 'w') as fb:
                fb.write('\n'.join([resource_id, name]))

        return resource_id
