import json
import logging
import os
import shutil
import subprocess

import requests
from flask import current_app as app
from flask_script import Command, Option
from sandbox_common.proxy import OAuth
from sandbox_common.rest import Client

from jafar.storages.memmap.storage import restore_dump

logger = logging.getLogger(__name__)

QLOUD_JAFAR_ENVIRONMENT_URL = 'https://platform.yandex-team.ru/api/v1/environment/dump/yandex-phone.jafar.production'


class UpdateSnapshot(Command):

    option_list = (
        Option('--sandbox_resource', dest='sandbox_resource', default=None, action='store',
               help="Update from sandbox resourse id"),
        Option('--skynet_resource', dest='skynet_resource', default=None, action='store', help="Update from torrent resourse id"),
        Option('--file', dest='file_path', default=None, action='store', help="Update from file"),
        Option('--lazy', dest='lazy', default=False, action='store_true', help="Don't unpack snapshot if it's already installed"),
    )

    def try_get_skynet_resource(self):
        distribution_path = '/etc/jafar-snapshot/jafar-snapshot.json'
        if os.path.exists(distribution_path):
            return json.load(open(distribution_path))['skynet_resource']
        dpkg = subprocess.Popen(('dpkg', '-l'), stdout=subprocess.PIPE)
        for line in dpkg.communicate()[0].split('\n'):
            if 'yandex-mobile-jafar-snapshot' in line:
                raise IOError("yandex-mobile-jafar-snapshot seems to be installed, but no {} found. Someting's broken with snapshot package".format(distribution_path))
        raise IOError("yandex-mobile-jafar-snapshot is not installed, cannot update snapshot")

    def download_file(self, skynet_resource):
        logger.info('Downloading resource {} from Skynet'.format(skynet_resource))
        folder_path = '/tmp/jafar-snapshot'
        try:
            if os.path.isdir(folder_path):
                shutil.rmtree(folder_path)
            else:
                os.remove(folder_path)
        except OSError:
            pass
        os.mkdir(folder_path)
        subprocess.check_call(['sky', 'get', '-u', '-d', folder_path, '-w', skynet_resource])
        logger.info('Successfully downloaded')

        # get file path ignoring any intermediate folders
        for root, dirs, files in os.walk(folder_path, topdown=False):
            return os.path.join(root, files[0])

    def update_from_sandbox(self, sandbox_resource):
        logger.info('Updating snapshot from Sandbox resource')
        client = Client(auth=OAuth(app.config['SANDBOX_OAUTH_TOKEN']))
        skynet_resource = client.resource[sandbox_resource].read()['skynet_id']
        return self.download_file(skynet_resource)

    def update_with_last(self):
        logger.info('Updating with last deployed snapshot')
        dump = requests.get(
            QLOUD_JAFAR_ENVIRONMENT_URL,
            headers={'Authorization': 'OAuth {}'.format(app.config['QLOUD_OAUTH_TOKEN'])}
        ).json()
        component = next(x for x in dump['components'] if x['componentName'] == 'recommender')
        resource = component['sandboxResources'][0]['id']
        return self.update_from_sandbox(resource)

    def run(self, sandbox_resource, skynet_resource, file_path, lazy):
        # only one of three options have to be enabled
        if sum([sandbox_resource is not None, skynet_resource is not None, file_path is not None]) > 1:
            raise ValueError('Leave params empty or choose one of the options: '
                             'sandbox_resource, skynet_resource or file')

        if sandbox_resource:
            path = self.update_from_sandbox(sandbox_resource)
        elif skynet_resource:
            # manual update: reserved when we know a torrent-file beforehand
            path = self.download_file(skynet_resource)
        elif file_path:
            path = file_path
        else:
            path = self.update_with_last()

        restore_dump(path, lazy=lazy)
