import logging
from operator import itemgetter

import numpy as np
from flask import current_app as app
from nile.api.v1 import extractors as ne, filters as nf
from qb2.api.v1 import filters as qf
from qb2.api.v1.typing import Yson

from jafar.datasets.base import BaseDatasetProcessor, get_table_name
from jafar.datasets.filters import append_value
from jafar.utils.io import get_cluster, get_apps_table
from jafar_yt.profile_dataframe_converter import UserProfileConverter
from jafar_yt.update_datasets import AdvisorMongoInstallMapper, AdvisorMongoUserFeatureMapper

logger = logging.getLogger(__name__)


def np_dtype_to_base(dtype):
    '''Replace numpy dtypes with basic'''
    dtypes_map = {np.object: Yson, np.bool: bool, np.ndarray: list}
    result = []
    for column, type_ in dtype:
        if type_ in dtypes_map:
            result.append((column, dtypes_map[type_]))
        else:
            result.append((column, type(type_(0).item())))  # convert numeric dtypes to native python

    return result


class AdvisorMongoDatasetProcessor(BaseDatasetProcessor):
    """
    Advisor's `packages_info`, aka currently installed apps
    pros: lots of data, cons: doesn't include user's history of app install
    """
    source = 'advisor_mongo'
    filters = BaseDatasetProcessor.filters + (append_value,)
    item_features_dtype = [
        ('item', np.object),
        ('category', 'S40'),
        ('publisher', 'U150'),
        ('title', 'U150'),
    ]

    def __init__(self, *args, **kwargs):
        super(AdvisorMongoDatasetProcessor, self).__init__(*args, **kwargs)
        self.yt_table_source = app.config['YT_PATH_USERS_FULL']
        self.yt_table_user_features = self.yt_table_result + '_user_features'
        self.yt_table_item_features = self.yt_table_result + '_item_features'
        self.interactions_mapper = AdvisorMongoInstallMapper

    def get_user_features(self, country, features=None):
        if features is not None:
            dtype = [(name, type_)
                     for name, type_ in UserProfileConverter.get_user_features_dtype()
                     if name in ['user'] + list(features)]
        else:
            dtype = UserProfileConverter.get_user_features_dtype()

        return self._download_table(get_table_name(self.yt_table_user_features, country), dtype)

    def get_item_features(self, country, features=None):
        if features is not None:
            dtype = [(name, type_)
                     for name, type_ in self.item_features_dtype
                     if name in ['item'] + list(features)]
        else:
            dtype = self.item_features_dtype

        return self._download_table(get_table_name(self.yt_table_item_features, country), dtype)

    @property
    def interactions_schema(self):
        return dict(np_dtype_to_base(UserProfileConverter.get_installs_dtype()) + [('value', float)])

    @property
    def interactions_dtype(self):
        return UserProfileConverter.get_installs_dtype() + [('value', np.float32)]

    def update_user_features(self, country):
        """
        This dataset includes additional frame with user features
        """
        logger.info("Updating %s user features for country %s", self.source, country)
        cluster = get_cluster()
        job = cluster.job()

        MAX_UPDATED_AGE = app.config['MAX_UPDATED_AGE']

        stream = job.table(
            self.yt_table_source
        ).map(
            AdvisorMongoUserFeatureMapper(country=country),
        ).filter(
            nf.custom(lambda x: x < MAX_UPDATED_AGE, 'updated_age')
        )

        schema = dict(np_dtype_to_base(UserProfileConverter.get_user_features_dtype()))

        stream.put(get_table_name(self.yt_table_user_features, country),
                   schema=schema)
        job.run()

    def update_item_features(self, country):
        """
        This dataset includes additional frame with item features
        """
        logger.info("Updating %s item features", self.source)
        cluster = get_cluster()
        job = cluster.job()

        region = app.config['GOOGLE_PLAY_REGIONS_MAP'][country]
        languages = app.config['EXPECTED_LANGUAGES_MAP'][country]

        job.table(
            get_apps_table()
        ).filter(
            qf.equals('region', region),
            qf.one_of('detected_lang', languages),
            qf.defined('subcategory_eng'),
            qf.nonzero('subcategory_eng'),
        ).project(
            item='id',
            # ignore FAMILY_* categories
            category=ne.custom(lambda cats: [cat for cat in cats if 'FAMILY_' not in cat][0], 'subcategory_eng'),
            title='name',
            publisher=ne.custom(itemgetter('name'), 'author'),
        ).put(
            path=get_table_name(self.yt_table_item_features, country),
            schema=dict(item=str, category=str, publisher=unicode, title=unicode),
        )
        job.run()

    def update(self, country):
        self.update_interactions(country)
        self.update_user_features(country)
        self.update_item_features(country)

