import logging
from datetime import date

import numpy as np
import yt.wrapper as yt
from flask import current_app as app
from nile.api.v1 import (
    aggregators as na,
    extractors as ne,
    filters as nf,
)
from qb2.api.v1 import filters as qf
from qb2.api.v1 import typing as qt
from yt import yson

from jafar.datasets import AdvisorMongoDatasetProcessor
from jafar.datasets.base import get_table_name
from jafar.utils.dyntables import dynamize_static_and_cleanup, create_dyntable
from jafar.utils.io import get_cluster
from jafar_yt.update_datasets import AdvisorMongoInstallMapper
from jafar_yt.utils.helpers import day_before, place_launched, EventValueMapperSelective, unify_uuid

logger = logging.getLogger(__name__)

ARRANGER_DTYPE = [
    ('user', np.object),
    ('item', np.object),
    ('value', np.int32),
]

TOP_CLASS_SCHEMA = yson.YsonList([
    {"name": "package_name", "type": "utf8", "sort_order": "ascending"},
    {"name": "class_name", "type": "utf8"},
])
TOP_CLASS_SCHEMA.attributes["unique_keys"] = True
TOP_CLASS_MODE = 'uncompressed'


class ArrangerLaunchesDatasetProcessor(AdvisorMongoDatasetProcessor):
    source = 'arranger'
    interactions_dtype = ARRANGER_DTYPE
    item_features_dtype = [
        ('item', np.object),
        ('category', 'S40'),
    ]

    @staticmethod
    def prepare_launches(job):
        day_from = day_before(app.config['ARRANGER_DAYS']).isoformat()
        day_to = day_before(1).isoformat()

        return job.table(
            yt.ypath_join(app.config['YT_METRIKA_PATH_1_DAY'], '{%s..%s}' % (day_from, day_to))
        ).filter(
            nf.equals('APIKey', app.config['YT_LAUNCHER_API_KEY']),
            nf.equals('EventName', 'app_launch'),
            qf.contains('Clids_Values', '2247990')
        ).map(
            EventValueMapperSelective(dict(packageName=qt.Optional[qt.Unicode],
                                           className=qt.Optional[qt.Unicode],
                                           action=qt.Optional[qt.Unicode],
                                           place=qt.Yson))
        ).filter(
            nf.custom(lambda x: (x or 'run') == 'run', 'action'),
            qf.defined('place', 'DeviceID', 'packageName', 'className')
        ).project(
            'className',
            user=ne.custom(unify_uuid, 'DeviceID').add_hints(type=qt.String),
            item='packageName',
            place=ne.custom(place_launched, 'place').add_hints(type=qt.SizedTuple[qt.String, qt.String])
        ).filter(
            qf.not_(qf.equals('place', ('homescreens', 'dock'))),
            qf.defined('user')
        ).groupby(
            'user', 'item', 'className'
        ).aggregate(
            value=na.count()
        )

    def update_interactions(self, country):
        job = get_cluster(backend='yql').job()
        top_class_name_path = yt.ypath_join(app.config['ARRANGER_TOP_CLASS_STATIC'], date.today().isoformat())
        top_class_name_path = get_table_name(top_class_name_path, country)

        if yt.exists(top_class_name_path):
            yt.remove(top_class_name_path)

        create_dyntable(top_class_name_path, yt, schema=TOP_CLASS_SCHEMA, in_memory_mode=TOP_CLASS_MODE)

        launches_counts = self.prepare_launches(job)

        # select most common className for packageName
        top_class_name = launches_counts.groupby(
            'item',
            'className'
        ).aggregate(
             value=na.sum('value')
        ).groupby(
            'item'
        ).aggregate(
            className=na.last('className', 'value')
        )

        # get installs from AdvisorMongoDatasetProcessor
        basket = job.table(
            app.config['YT_PATH_USERS_FULL']
        ).map(
            AdvisorMongoInstallMapper(country, user_apps_only=False),
            intensity='cpu'
        ).project(
            item=ne.custom(lambda x: x, 'item').add_hints(type=qt.String),
            user=ne.custom(lambda x: x, 'user').add_hints(type=qt.String)
        )

        # use only selected classNames
        launches_counts = launches_counts.join(
            top_class_name,
            by=['item', 'className'],
            assume_unique_right=True,
            assume_small_right=True
        )

        # add not launched items
        zero_launches = basket.join(
            launches_counts,
            by=['user', 'item'],
            type='left_only',
            assume_unique=True
        ).project(
            'user',
            item_left='item'
        ).join(
            launches_counts.unique('user'),  # only users from launches are allowed
            by='user',
            type='inner',
            assume_small_right=True,
            assume_unique_right=True
        ).project(
            'user',
            item='item_left',
            value=ne.const(0)
        ).join(
            top_class_name,
            by='item',
            type='inner',
            assume_small_right=True,
            assume_unique_right=True
        )

        job.concat(
            zero_launches,
            launches_counts
        ).project(
            'user',
            'value',
            item=ne.custom(u'{}/{}'.format, 'item', 'className').add_hints(type=qt.String)
        ).put(
            get_table_name(self.yt_table_result, country)
        )

        top_class_name.project(
            package_name='item',
            class_name='className'
        ).sort(
            'package_name'
        ).put(
            top_class_name_path + "_temp"
        )

        job.run()

        # merge temp table with empty table for unique_keys = True
        # and reduce chunk size
        yt.run_merge(
            source_table=[top_class_name_path + "_temp",
                          top_class_name_path],
            destination_table=top_class_name_path,
            spec=dict(
                force_transform=True,
                combine_chunks=True,
                job_io=dict(
                    table_writer=dict(
                        block_size=256 * 2 ** 10,
                        desired_chunk_size=100 * 2 ** 20
                    )))
        )
        yt.remove(top_class_name_path + "_temp")

        return top_class_name_path

    def update_item_features(self, country, classname_table):
        job = get_cluster(backend='yql').job()
        job.table(
            get_table_name(AdvisorMongoDatasetProcessor().yt_table_item_features, country)
        ).join(
            job.table(
                classname_table
            ).project(
                'class_name',
                item='package_name'
            ),
            by='item',
            assume_unique=True,
            type='right'
        ).project(
            ne.all(),
            item=ne.custom(u'{}/{}'.format, 'item', 'class_name').add_hints(type=qt.String)
        ).put(
            get_table_name(self.yt_table_item_features, country)
        )
        job.run()

    def update(self, country):
        table = self.update_interactions(country)
        self.update_item_features(country, table)
        kwargs_list = [dict(base_path=app.config['ARRANGER_TOP_CLASS_STATIC'],
                            ttl=app.config['ARRANGER_TOP_CLASS_STATIC_TTL'])]
        dynamize_static_and_cleanup(table=table,
                                    schema=TOP_CLASS_SCHEMA,
                                    mount=True,
                                    symlink=app.config['ARRANGER_TOP_CLASS_DYNAMIC'][country],
                                    cleanup_kwargs_list=kwargs_list,
                                    in_memory_mode=TOP_CLASS_MODE)
