import logging
from datetime import date, timedelta

import yt.wrapper as yt
from flask import current_app as app
from nile.api.v1 import filters as nf
from qb2.api.v1 import filters as qf

from jafar.datasets.base import BaseDatasetProcessor, get_table_name
from jafar.utils.io import get_cluster
from jafar_yt import update_events as yt_operations
from jafar_yt.update_datasets import (
    ConversionsNoPromoMapper,
    ConversionsPromoMapper
)
from jafar_yt.update_events import REC_EVENTS

logger = logging.getLogger(__name__)

EVENTS_AGGREGATE_DAYS = 7


class ConversionsDatasetProcessor(BaseDatasetProcessor):
    def update_interactions(self, country):
        # find latest existing table
        day_to = date.today()
        while not yt.exists(yt.ypath_join(app.config['YT_METRIKA_PATH_1_DAY'], day_to.isoformat())):
            day_to -= timedelta(days=1)

        day_from = (day_to - timedelta(days=EVENTS_AGGREGATE_DAYS))

        job = get_cluster().job()
        stream = job.table(
            yt.ypath_join(app.config['YT_METRIKA_PATH_1_DAY'], '{%s..%s}' % (day_from, day_to))
        ).filter(
            nf.equals('APIKey', app.config['YT_LAUNCHER_API_KEY']),
            qf.one_of('EventName', REC_EVENTS),
        ).map(
            yt_operations.MapperEvents(ignore=0)
        ).groupby(
            'user', 'item'
        ).reduce(
            yt_operations.TargetReducer(),
            memory_limit=8 * 1024
        ).sort(
            'user', 'item', 'timestamp'
        ).map(
            self.interactions_mapper(country)
        )
        stream.put(get_table_name(self.yt_table_result, country), schema=self.interactions_schema)
        job.run()


class ConversionsNoPromoDatasetProcessor(ConversionsDatasetProcessor):
    """
    `conversion` events (rec_view followed by rec_install + some heuristics)
    collected from Advisor's metrika
    experimental dataset, work in progress
    """
    source = 'conversions_no_promo'
    interactions_mapper = ConversionsNoPromoMapper
    filters = []


class ConversionsPromoDatasetProcessor(ConversionsDatasetProcessor):
    """
    `conversion` events (rec_view followed by rec_install + some heuristics)
    collected from Advisor's metrika
    experimental dataset, work in progress
    """
    source = 'conversions_promo'
    interactions_mapper = ConversionsPromoMapper
    filters = []

    @property
    def interactions_schema(self):
        return dict(click=int, item=str, placement=str, user=str, value=int, view_count=int)
