import logging
from datetime import datetime, timedelta

from flask import current_app as app

from jafar import filters
from jafar.exceptions import FeedException

logger = logging.getLogger(__name__)


class FeedBlock(object):
    def __init__(self, items, title, subtitle, algorithm, card_type, content_type,
                 explanation=None, reserved=None, rotation_interval=None,
                 placement_id=None, external_promo_provider=None, external_promo_ttl=None):
        self.items = items
        self.title = title
        self.subtitle = subtitle
        self.algorithm = algorithm
        self.card_type = card_type
        self.content_type = content_type
        self.explanation = explanation
        self.reserved = reserved or []
        self.rotation_interval = rotation_interval
        self.placement_id = placement_id
        self.external_promo_provider = external_promo_provider
        self.external_promo_ttl = external_promo_ttl


class FeedPage(object):
    def __init__(self, blocks, expire_at, generated_at=None):
        self.blocks = blocks
        self.expire_at = expire_at
        self.generated_at = generated_at or datetime.utcnow()

    @property
    def items(self):
        for block in self.blocks:
            for item in block.items:
                yield item['package_name']


class FeedMeta(object):
    def __init__(self, expire_at, page_count=0, item_filter=None, item_filter_size=2,
                 explanation_filter=None, explanation_filter_size=5, generated_at=None):
        self.page_count = page_count
        self.item_filter = (
                item_filter or
                filters.MultipleBloomItemFilter(item_filter_size)
        )
        self.explanation_filter = (
                explanation_filter or
                filters.MultipleBloomItemFilter(explanation_filter_size)
        )
        self.generated_at = generated_at or datetime.utcnow()
        self.expire_at = expire_at

    def update_item_filter(self, items):
        for item in items:
            if item not in self.item_filter:
                self.item_filter.add(item)
                logger.debug('Item %s is stored in Bloom filter', item)

    def update_explanation_filter(self, explanations):
        for explanation in explanations:
            if explanation not in self.explanation_filter:
                self.explanation_filter.add(explanation)
                logger.debug('Explanation %s is stored in Bloom filter', explanation)


class Feed(object):
    def __init__(self, user, strategy, recommenders, backend_type, cache_key,
                 experiment_name, block_count, lifetime_seconds, item_filter_size,
                 explanation_filter_size, promo_provider=None, promo_provider_kwargs=None,
                 categories=None, place=None, editor_content=None, promo_placeholders=False):
        """
        :param user: `jafar.models.user.User` object
        :param strategy: an instance of `jafar.feed.strategy.Strategy`
        :param recommenders: collection of (FeedRecommender class, kwargs) tuples
        :param backend_type: a class from `jafar.feed.backend`
        :param cache_key: string to retrieve previously cached feed pages
        :param experiment_name: string
        :param block_count: how many FeedBlocks fit in one FeedPage
        :param lifetime_seconds: feed cache expiration time
        :param item_filter_size: between two occurrences of the same recommended item
                                 must be at least this many pages
        :param explanation_filter_size: between two occurrences of the same FeedBlock
                                        (having the same `explanation` property) must
                                        be at least this many pages
        :param promo_provider: a class from `jafar.data_providers.promo`
        :param promo_provider_kwargs: intialization params for promo provider
        :param categories: restrict recommendations by this app categories only
        :param place: request recommendations for a particular placement (Zen/Gamehub/etc)
        :param editor_content: a dictionary of {Editor card model: number of cards per page}
        :param promo_placeholders: does client support promo placeholders for direct

        :returns:
        """
        if not recommenders and not editor_content:
            raise FeedException('Need at least one feed recommender or editor content specified')
        self.user = user
        self.strategy = strategy
        self.categories = categories
        self.place = place
        self.recommenders = self.init_recommenders(recommenders)
        self.experiment_name = experiment_name
        self.editor_content = editor_content or {}
        self.editor_content_block_count = sum(self.editor_content.values())
        self.block_count = block_count - self.editor_content_block_count
        self.backend = backend_type(self, lifetime_seconds)
        self.cache_key = cache_key
        self.item_filter_size = item_filter_size
        self.explanation_filter_size = explanation_filter_size
        self.meta = self.prepare_meta(lifetime_seconds)
        self.item_filter = self.meta.item_filter
        self.promo_provider = promo_provider(**promo_provider_kwargs) if promo_provider else None
        self.promo_placeholders = promo_placeholders

    @property
    def is_persistent(self):
        if app.config['READ_ONLY']:
            return False
        return self.cache_key is not None

    def prepare_meta(self, lifetime_seconds):
        if self.is_persistent:
            meta = self.backend.load_meta()
            if meta:
                return meta
        now = datetime.utcnow()
        return FeedMeta(
            item_filter_size=self.item_filter_size,
            explanation_filter_size=self.explanation_filter_size,
            generated_at=now,
            expire_at=now + timedelta(seconds=lifetime_seconds)
        )

    def init_recommenders(self, recommenders):
        # expecting `recommenders` to be the list of
        # (`recommender_class`, `params`) tuples
        # NOTE: make it more obvious
        result = []
        for recommender_class, params in recommenders:
            params = params or {}
            params.update(categories=self.categories)
            params.update(place=self.place)
            result.append(recommender_class(**params))
        return result

    def generate_new_page(self, page_number):
        previous_page = self.backend.load_page(page_number - 1) if page_number > 0 else None
        # insert editor blocks in random positions
        blocks = self.strategy.get_blocks(feed=self, previous_page=previous_page)
        page = FeedPage(blocks=blocks, expire_at=self.meta.expire_at)
        if self.is_persistent:
            for block in blocks:
                self.update_meta(block)

            self.backend.save_page(page)
            self.meta.page_count += 1
            self.backend.save_meta(self.meta)
        return page

    def get_page(self, page_number=1):
        if self.is_persistent and page_number <= self.meta.page_count:
            logger.debug('Loading page %s from backend', page_number)
            page = self.backend.load_page(page_number)
            if page:
                return page
            logger.warn(
                "Feed meta reported %s pages, but couldn't fetch page %s",
                self.meta.page_count, page_number
            )
        logger.debug('Generating new page')
        return self.generate_new_page(page_number)

    def update_meta(self, block):
        self.meta.update_item_filter([
            item['package_name'] for item in block.items if 'package_name' in item
        ])
        self.meta.update_explanation_filter([block.explanation])
