import logging
import random
from collections import defaultdict
from itertools import cycle, tee

import numpy as np
from flask import current_app as app

from jafar import storage_wrapper
from jafar.feed.base import FeedBlock
from jafar.pipelines import ids
from jafar.pipelines.pipeline import PipelineConfig
from jafar.pipelines.predefined import (all_predefined_pipelines,
                                        predefined_arranger_pipelines)
from jafar.utils.iter import consume, take
from jafar.utils.structarrays import DataFrame

logger = logging.getLogger(__name__)


class GreedyGroupDistributorMixin(object):
    """Trying to fill first cards as possible"""

    def _split_into_blocks(self, cards):
        lower_bounds = [c.min_count for c in cards]
        upper_bounds = [c.max_count for c in cards]

        residual = self.size - sum(lower_bounds)
        assert residual >= 0, 'Not enough recommendations for all cards'
        i = 0
        while residual > 0 and i < len(lower_bounds):
            if upper_bounds[i] is None:
                lower_bounds[i] += residual
                break

            additional = min(upper_bounds[i] - lower_bounds[i], residual)
            lower_bounds[i] += additional
            residual -= additional
            i += 1

        return lower_bounds


class RecommendationGroup(GreedyGroupDistributorMixin):
    """
    Wrapper around a dataframe which defines two iterators
    to read recommendations and reserve items at the same time.
    """
    content_type = 'apps'

    def __init__(self, explanation, recommender, recommendations, weight):
        self.explanation = explanation
        self.recommender = recommender
        self.recommendations = recommendations
        self.weight = weight
        self.init_iterators()

    @property
    def size(self):
        return len(self.recommendations)

    def init_iterators(self):
        self.item_iter, self.reserve_iter = tee(self.recommendations, 2)
        self.reserve_iter = cycle(self.reserve_iter)

    def read_item_iter(self, count):
        # shift reserve iterator accordingly
        consume(self.reserve_iter, count)
        return take(self.item_iter, count)

    def read_reserve_iter(self, count):
        return take(self.reserve_iter, count)

    def filter(self, item_filter):
        """
        Filters recommendations by a set-like object and discards items which
        are already in it.
        """
        self.recommendations = [
            element for element in self.recommendations
            if element['package_name'] not in item_filter
        ]
        self.init_iterators()

    def update_filter(self, item_filter):
        """
        Updates item filter with items from recommendation group.
        """
        item_filter.update(element['package_name'] for element in self.recommendations)

    def get_blocks(self, cards):
        """
        Outputs a FeedBlock containing required amount of recommended items.
        """
        title, subtitle = self.recommender.get_title_pair(self.explanation)
        block_sizes = self._split_into_blocks(cards)
        blocks = []
        for size, card in zip(block_sizes, cards):
            blocks.append(FeedBlock(
                items=self.read_item_iter(size),
                reserved=self.read_reserve_iter(card.reserved_count),
                title=title,
                subtitle=subtitle,
                card_type=card.card_type,
                content_type=card.content_type,
                algorithm=self.recommender.experiment_name,
                explanation=self.explanation,
                rotation_interval=card.rotation_interval,
            ))
        return blocks


class PromoRecommendationGroup(RecommendationGroup):
    content_type = 'promo'


class PlaceholdersPromoRecommendationGroup(RecommendationGroup):
    content_type = 'promo'

    def init_iterators(self):
        self.iter = None
        self.reserve_iter = iter(self.recommendations)

    def read_item_iter(self, count):
        return [{'placeholder': True} for _ in xrange(count)]

    def filter(self, item_filter):
        """
        Do not filter apps for promo placeholders
        """
        pass

    def get_blocks(self, cards):
        blocks = super(PlaceholdersPromoRecommendationGroup, self).get_blocks(cards)
        for block in blocks:
            block.placement_id = self.recommender.placement_id
            block.external_promo_provider = self.recommender.external_promo_provider
            block.external_promo_ttl = self.recommender.external_promo_ttl
        return blocks


class MixedRecommendationGroup(RecommendationGroup):
    """
    Contains two distinct sets of recommendations: promo and non-promo items.
    Base class for PromoExtraRecommendationGroup and OldFashionedRecommendationGroup.

    Override read_item_iter and  init_iterators methods to determine a way of
    promo and non-promo recommendations being mixed
    """

    def __init__(self, promo=None, *args, **kwargs):
        self.promo = promo or []
        super(MixedRecommendationGroup, self).__init__(*args, **kwargs)

    def filter(self, item_filter):
        self.recommendations = [
            element for element in self.recommendations
            if element['package_name'] not in item_filter
        ]
        self.promo = [
            element for element in self.promo
            if element['package_name'] not in item_filter
        ]
        self.init_iterators()

    def update_filter(self, item_filter):
        item_filter.update(element['package_name'] for element in self.recommendations)
        item_filter.update(element['package_name'] for element in self.promo)

    def read_item_iter(self, count):
        raise NotImplementedError

    def init_iterators(self):
        self.item_iter, self.reserve_iter = tee(self.recommendations, 2)
        self.reserve_iter = cycle(self.reserve_iter)
        self.promo_iter = iter(self.promo)


class PromoExtraRecommendationGroup(MixedRecommendationGroup):
    """
    Adds some non-promo items to promo to assure that one smallest card in the feed will be filled in worst case.

    Is used by PromoExtraRecommender.
    """

    def __init__(self, min_card_size, *args, **kwargs):
        self.min_card_size = min_card_size
        super(PromoExtraRecommendationGroup, self).__init__(*args, **kwargs)

    @property
    def size(self):
        return max(len(self.promo), self.min_card_size)

    def read_item_iter(self, count):
        promo_chunk = take(self.promo_iter, count)
        non_promo_count = count - len(promo_chunk)
        non_promo_chunk = take(self.item_iter, non_promo_count)
        consume(self.reserve_iter, len(non_promo_chunk))
        return promo_chunk + non_promo_chunk


class FeedRecommender(object):
    """
    Creates prediction from the pipeline without Experiments & Postprocessors API
    """
    pipeline_config = PipelineConfig(
        recommendation_mode='generate',
        online=True,
    )
    recommendation_group_class = RecommendationGroup

    def __init__(self, title_pairs, experiment_name='default',
                 categories=None, default_categories=None,
                 place=None, weight=0.0,
                 top_n=None):
        self.title_pairs = title_pairs
        self.experiment_name = experiment_name
        self.categories = categories
        self.default_categories = default_categories
        self.place = place
        self.weight = weight
        self.top_n = top_n or app.config['TOP_N_COUNT']

    def get_title_pair(self, explanation):
        pair = random.choice(self.title_pairs)
        return pair['title'], pair['subtitle']

    def prepare_recommendations(self, predictions):
        sort_order = ('priority', 'value') if 'priority' in predictions.dtype.names else ('value',)
        predictions = predictions.drop_columns(['user'])
        predictions.sort(order=sort_order)
        predictions = predictions[::-1]

        if 'offer_id' in predictions.dtype.names:
            extra_columns = ['offer_id', 'expected_fee', 'cpm', 'mark_sponsored']
        else:
            extra_columns = []

        columns_mapping = {
            'item': 'package_name',
            'value': 'score'
        }

        columns = list(set(columns_mapping.keys() + extra_columns).intersection(set(predictions.columns)))
        predictions = predictions[columns]
        if 'mark_sponsored' not in predictions:  # don't mark apps as sponsored by default
            predictions = predictions.append_column(np.zeros(predictions.shape[0], dtype=np.bool), 'mark_sponsored')
        predictions.rename_columns(columns_mapping)

        result = predictions.to_list_of_dicts()
        if 'offer_id' in columns:
            for prediction in result:
                if 'offer_id' in prediction and prediction['offer_id'] is None:
                    del prediction['offer_id']
        for app in result:
            app['popup_type'] = 'App_popup'
        return result

    def create_pipeline(self, feed):
        return self.pipeline_creator(
            pipeline_config=self.pipeline_config,
            storage=storage_wrapper.storage,
            top_n=self.top_n
        )

    def get_target_frame(self, feed):
        """
        Returns a FRAME_KEY_TARGET ("query") frame for pipeline.
        """
        if self.place:
            # for the case when we use 'placement' to properly calculate CPM
            return DataFrame.from_dict({
                'user': [str(feed.user.device_id)],
                'placement': [self.place]
            })
        else:
            return DataFrame.from_dict({'user': [str(feed.user.device_id)]})

    def get_recommendations_from_pipeline(self, feed, pipeline):
        context = pipeline.create_initial_context(
            requested_categories=self.categories,
            default_categories=self.default_categories,
            frames={
                ids.FRAME_KEY_TARGET: self.get_target_frame(feed),
            }
        )
        if feed.user.clids:
            clid_names, clid_values = zip(*feed.user.clids.iteritems())
            context.data[ids.FRAME_KEY_CLIDS] = DataFrame.from_dict({'name': clid_names, 'value': clid_values})
        return pipeline.predict_top_n(context)

    def format_explanation(self, explanation):
        return None

    def split_recommendations_into_groups(self, recommendations):
        yield None, recommendations

    def get_recommendation_groups(self, feed, cards):
        pipeline = self.create_pipeline(feed)
        recommendations = self.get_recommendations_from_pipeline(feed, pipeline)
        if len(recommendations) == 0:
            logger.info("Couldn't get recommendations: %s", self.experiment_name)

        recommendation_groups = []
        for explanation, frame in self.split_recommendations_into_groups(recommendations):
            recommendation_groups.append(self.recommendation_group_class(
                self.format_explanation(explanation), self, self.prepare_recommendations(frame),
                weight=self.weight
            ))

        return recommendation_groups


class ArrangerRecommender(FeedRecommender):
    def __init__(self, user, pipeline, used_apps=None, rearrangement_ranges=None, top_n=None):
        self.user = user
        self.pipeline_creator = predefined_arranger_pipelines[pipeline]
        self.top_n = top_n
        self.weight = 0
        self.rearrangement_ranges = rearrangement_ranges
        self.used_apps = used_apps

    def create_pipeline(self, feed=None):
        return self.pipeline_creator(
            pipeline_config=self.pipeline_config,
            storage=storage_wrapper.storage,
            top_n=self.top_n
        )

    def get_target_frame(self, feed=None):
        return DataFrame.from_dict({'item': self.user.packages,
                                    'user': [str(self.user.device_id)] * len(self.user.packages)})

    def get_range_frame(self):
        left, right = zip(*self.rearrangement_ranges)
        return DataFrame.from_dict({'left': left,
                                    'right': right})

    def get_used_apps_frame(self):
        return DataFrame.from_dict({'item': self.used_apps,
                                    'user': [str(self.user.device_id)] * len(self.used_apps)})

    def get_recommendations_from_pipeline(self, feed, pipeline):
        context = pipeline.create_initial_context(
            frames={
                ids.FRAME_KEY_TARGET: self.get_target_frame(),
            },
            country='RU'  # TODO: remove this and take users country
        )
        if self.rearrangement_ranges:
            context.data[ids.FRAME_KEY_RANGE] = self.get_range_frame()
        if self.used_apps:
            context.data[ids.FRAME_DEFAULT_ITEMS] = self.get_used_apps_frame()
        return pipeline.predict_top_n(context)

    def prepare_recommendations(self, predictions):
        predictions.sort(order=('value',))
        return predictions[::-1]['item']


class PlaceholdersPromoFeedRecommender(FeedRecommender):
    recommendation_group_class = PlaceholdersPromoRecommendationGroup

    def __init__(self, pipeline, placement_id, external_promo_provider, external_promo_ttl, *args, **kwargs):
        super(PlaceholdersPromoFeedRecommender, self).__init__(*args, **kwargs)
        self.pipeline = pipeline
        self.pipeline_creator = all_predefined_pipelines[pipeline]
        self.placement_id = placement_id
        self.external_promo_provider = external_promo_provider
        self.external_promo_ttl = external_promo_ttl

    def get_recommendation_groups(self, feed, cards):
        """ Do not return placeholders groups if client does not support placeholders format """
        if not feed.promo_placeholders:
            return []
        return super(PlaceholdersPromoFeedRecommender, self).get_recommendation_groups(feed, cards)


class ParametrizedFeedRecommender(FeedRecommender):
    pipeline_choices = None

    def __init__(self, pipeline, *args, **kwargs):
        """
        Unlike other FeedRecommenders, this one can be parametrized with the name of the pipeline.
        """
        if self.pipeline_choices is not None:
            assert pipeline in self.pipeline_choices, \
                '{} only allows specific pipelines: {}'.format(self.__class__.__name__, self.pipeline_choices)
        self.pipeline_creator = all_predefined_pipelines[pipeline]
        self.pipeline_name = pipeline
        super(ParametrizedFeedRecommender, self).__init__(*args, **kwargs)


class SimpleLocalFeedRecommender(ParametrizedFeedRecommender):
    """
    Outputs city-specific FeedBlocks, including city as
    explanation, corresponding title/subtitle and a
    background image.

    NOTE: recommendations themselves are not necessarily
    produced by "local" pipeline: though it is used by
    default, a different pipeline can be specified (for example,
    when we don't have enough city-specific apps but still want
    a feed block to look personalized, even if it's really not).
    """
    missing_region_id = -1

    def __init__(self, pipeline='local', *args, **kwargs):
        super(SimpleLocalFeedRecommender, self).__init__(pipeline, *args, **kwargs)

    def format_explanation(self, explanation):
        return {'region_id': explanation}

    def split_recommendations_into_groups(self, recommendations):
        region_id = int(recommendations['lbs_region_city'][0]) if len(recommendations) > 0 else self.missing_region_id
        yield region_id, recommendations


class SimilarItemsRecommender(ParametrizedFeedRecommender):
    def format_explanation(self, explanation):
        return {'similar_to': explanation}

    def split_recommendations_into_groups(self, recommendations):
        for neighbor, frame in recommendations.groupby('similar_to'):
            yield neighbor, frame


# NOTE: it doesn't have to be a class other than for naming convenience
class TopNRecommender(ParametrizedFeedRecommender):
    pass


class TopNCategorizedRecommender(ParametrizedFeedRecommender):
    def format_explanation(self, explanation):
        return {'category': explanation}

    def split_recommendations_into_groups(self, recommendations):
        for category, frame in recommendations.groupby('category'):
            yield category, frame


class GiftsRecommendationGroup(RecommendationGroup):
    content_type = 'gifts'


class GiftsRecommender(ParametrizedFeedRecommender):
    """Recommender for gifts, not apps"""
    recommendation_group_class = GiftsRecommendationGroup

    def __init__(self, max_view_count=None, disliked_items_ttl=None, *args, **kwargs):
        super(GiftsRecommender, self).__init__(*args, **kwargs)
        self.pipeline_params = defaultdict(dict)
        if max_view_count is not None:
            self.pipeline_params['impression_discount']['max_view_count'] = max_view_count
        if disliked_items_ttl is not None:
            self.pipeline_params['filter_banned']['disliked_items_ttl'] = disliked_items_ttl

    def get_target_frame(self, feed):
        result = super(GiftsRecommender, self).get_target_frame(feed)
        return result.append_column([feed.user.passport_uid], 'passport_uid')

    def create_pipeline(self, feed):
        pipeline = super(GiftsRecommender, self).create_pipeline(feed)
        pipeline.set_params(self.pipeline_params)
        return pipeline

    def prepare_recommendations(self, predictions):
        recommendations = []
        for gift in predictions:
            result = {
                'package_name': gift['item'],
                'popup_type': 'Bonus_universal_popup',
            }

            # Optional fields
            fields = gift.dtype.fields
            if 'id' in fields:
                result['key'] = gift['id']
            if 'code' in fields:
                result['code'] = gift['code']
            if 'offer_id' in fields:
                result['offer_id'] = gift['offer_id']
            recommendations.append(result)
        return recommendations
