from collections import namedtuple
from uuid import UUID

from marshmallow import Schema, fields, post_load, validate, pre_load, post_dump

from jafar.feed.fields import CommaSeparatedList, ListField
from jafar.models.user import User

# NOTE: while we don't have fallback for feed, any missing users will be forced to use the default country
default_country = 'US'
DATETIME_FORMAT = '%Y-%m-%dT%H:%M:%S'


class FeedButton(Schema):
    text = fields.Str()
    url = fields.Url()


class FeedBlockSchema(Schema):
    items = fields.List(fields.Dict())
    title = fields.Str()
    subtitle = fields.Str()
    card_type = fields.Str()
    content_type = fields.Str()
    algorithm = fields.Str()
    explanation = fields.Dict()
    reserved = fields.List(fields.Dict())
    rotation_interval = fields.Int(allow_none=True)
    placement_id = fields.Str()
    external_promo_provider = fields.Str()
    external_promo_ttl = fields.Int()


class FeedPageSchema(Schema):
    blocks = fields.Nested(FeedBlockSchema, many=True)
    expire_at = fields.DateTime(required=True, format=DATETIME_FORMAT)


class CardConfigSchema(Schema):
    type = fields.Str(required=True, load_from='card_type')
    count = fields.Int(required=False, missing=None)


class UserInfoSchema(Schema):
    device_id = fields.UUID(required=True)
    card_configs = fields.Nested(CardConfigSchema, required=True, many=True, missing=None, allow_none=False)
    clids = fields.Dict(required=True)
    passport_uid = fields.Int(required=False)

    @post_load
    def to_object(self, data):
        return User(**data)


class FeedRequestSchema(Schema):
    PAGE_LIMIT = 100

    page = fields.Int(required=False, missing=1, validate=validate.Range(1, PAGE_LIMIT))
    place = fields.Str(allow_none=True, missing=None)
    user_info = fields.Nested(UserInfoSchema)
    google_categories = fields.List(fields.Str, required=False, missing=list, allow_none=True)
    promo_placeholders = fields.Bool(required=False, missing=False)
    cache_key = fields.Str(required=False, missing=None, allow_none=True)
    # Block counts override is used for old 1.x.x launchers that send required groups_count in request
    block_count_override = fields.Int(required=False, missing=None, allow_none=True)

    @post_load
    def to_object(self, data):
        if not data['google_categories']:
            data['google_categories'] = []
        data['categories'] = data['google_categories']
        del data['google_categories']
        return data


class RecConfigSchema(Schema):
    card_type = fields.Str()
    count_is_required = fields.Bool()


class SearchSchema(Schema):
    query = fields.Str(required=True)
    count = fields.Int(required=True, validate=validate.Range(1, 100))
    country = fields.Str(required=True)

    @pre_load
    def normalize_query(self, data):
        data = data.copy()
        data['query'] = data['query'].lower().strip()
        return data


class VangaSchema(Schema):
    device_id = fields.UUID(required=True)
    places_blacklist = CommaSeparatedList(fields.Str(validate=validate.Regexp('^.*\.?.*$')), required=False)
    version = fields.Integer(required=False)
    packages = ListField(fields.Str(), required=False)


class PostVangaSchema(VangaSchema):  # TODO: remove this after advisor deploy
    places_blacklist = ListField(fields.Str(validate=validate.Regexp('^.*\.?.*$')), required=False)


class VangaResponseSchema(Schema):
    stats = fields.Dict()
    lifetime_seconds = fields.Int()


ArrangerUserInfo = namedtuple('ArrangerUserInfo', 'device_id, packages, grid_width, grid_height, defaults')


DEFAULT_CATEGORIES = (
    'DIALER',
    'MESSAGING',
    'BROWSER',
    'CAMERA',
    'GALLERY',
    'CALENDAR',
    'CALCULATOR',
    'CLOCK',
    'SETTINGS',
)


class DefaultApplicationsSchema(Schema):
    category = fields.Integer(validate=validate.Range(0, len(DEFAULT_CATEGORIES) - 1))
    package = fields.String()


class ArrangerSchema(Schema):
    device_id = fields.UUID(required=False, missing=str(UUID(int=0)))
    packages = ListField(fields.Str(), required=True)
    grid_width = fields.Integer(required=True, validate=validate.Range(3, 10))
    grid_height = fields.Integer(required=True, validate=validate.Range(3, 10))
    defaults = ListField(fields.Nested(DefaultApplicationsSchema()), missing=list, required=False)

    @post_load
    def to_object(self, data):
        for obj in data['defaults']:
            obj['category'] = DEFAULT_CATEGORIES[obj['category']]
        return ArrangerUserInfo(**data)


class PointSchema(Schema):
    x = fields.Integer(required=True, validate=validate.Range(0))
    y = fields.Integer(required=True, validate=validate.Range(0))


class ScreenItemSchema(Schema):
    cell = fields.Nested(PointSchema(), required=True)
    span = fields.Nested(PointSchema(), required=True)
    value = fields.Str(required=True)
    item_type = fields.Str(required=True)


class ArrangementSchema(Schema):
    dock = ListField(fields.Nested(ScreenItemSchema()))
    screens = ListField(ListField(fields.Nested(ScreenItemSchema())))

    @post_dump
    def retrieve_dock(self, data, **kwargs):
        if data.get('screens'):
            data['dock'], data['screens'] = data['screens'][0], data['screens'][1:]
        return data
