import logging
from collections import defaultdict

import cvxopt
import numpy as np
from cvxopt.base import matrix
from cvxopt.glpk import ilp

from jafar.exceptions import RecommenderNotReady, FeedException
from jafar.utils import get_all_subclasses
from jafar.web.exceptions import NotFound

logger = logging.getLogger(__name__)

# silencing glpk output: https://groups.google.com/forum/#!topic/cvxopt/hXkwAqNPCy0
cvxopt.glpk.options['msg_lev'] = 'GLP_MSG_OFF'


class Problem(object):
    """
    Contains matrices and right-hand sides for an integer programming problem.
    """

    def __init__(self, n_blocks, cards, recommendation_groups, card_weights,
                 group_weights, randomization, previous_page=None):
        self.n_blocks = n_blocks
        self.n_recommendation_groups = len(recommendation_groups)
        self.n_cards = len(cards)
        self.card_sizes = np.array([card.size for card in cards])
        self.recommendation_groups = recommendation_groups
        self.cards = cards
        self.card_weights = card_weights
        self.group_weights = group_weights
        self.randomization = randomization
        self.previous_page = previous_page
        # NOTE: constraint matrix notation is a bit ambiguous, but
        # we're keeping it this way to maintain correspondence with cvxopt
        self.G = [np.zeros(self.dimensions).ravel()]  # inequality constraint matrix (left-hand side)
        self.h = [0]  # inequality constraint right-hand side
        self.A = [np.zeros(self.dimensions).ravel()]  # equality constraint matrix (left-hand side)
        self.b = [0]  # equality constraint right-hand side

    @property
    def dimensions(self):
        return (self.n_blocks, self.n_recommendation_groups, self.n_cards)

    def get_scores(self):
        # cvxopt needs double precision (https://stackoverflow.com/questions/33423081/converting-numpy-vector-to-cvxopt)
        scores = np.ones(self.dimensions, dtype=np.float64)
        scores *= np.exp(-np.arange(self.dimensions[0]))[:, None, None]
        scores *= self.group_weights[None, :, None]
        scores *= self.card_weights[None, None, :]
        scores *= np.random.uniform(low=0.9, high=1.1, size=self.dimensions) ** self.randomization

        # we're solving minimization problem, hence subtraction
        return -scores.ravel()

    def solve(self):
        G = np.vstack(self.G).astype(np.float64)
        h = np.hstack(self.h).reshape(G.shape[0], 1).astype(np.float64)
        A = np.vstack(self.A).astype(np.float64)
        b = np.vstack(self.b).reshape(A.shape[0], 1).astype(np.float64)
        n_vars = np.product(self.dimensions)

        # coefficients for objective function, solution would be found in a way to maximize sum of these
        scores = self.get_scores()

        status, solution = ilp(
            matrix(scores),
            matrix(G),
            matrix(h),
            matrix(A),
            matrix(b),
            I=set(),
            B=set(xrange(n_vars))
        )
        if status != 'optimal':
            raise FeedException("Failed to find a solution for an integer programming problem: {}".format(status))
        return np.array(solution).flatten()


class Constraint(object):
    def apply(self, problem):
        raise NotImplementedError


class CustomConstraint(Constraint):
    """
    Constraint that can be enabled or disabled from admin interface
    """
    hint = None


class MandatoryConstraint(Constraint):
    """
    Constraint that will be applied in any case
    """
    pass


class OneHotPositionConstraint(MandatoryConstraint):
    """
    Must have exactly one recommendation group and one card for each block.
    """

    def apply(self, problem):
        coefs = np.zeros(problem.dimensions, dtype=np.int32)  # allocate memory once
        for i in xrange(problem.n_blocks):
            coefs[i, :, :] = 1
            problem.A.append(coefs.flatten())
            problem.b.append(1)
            coefs[i, :, :] = 0


class RecommendationGroupCapacityConstraint(MandatoryConstraint):
    """
    Sum over blocks for each recommendation group weighted by card sizes must be less or equal
    to group size.
    """

    def apply(self, problem):
        coefs = np.zeros(problem.dimensions, dtype=np.int32)
        for j in xrange(problem.n_recommendation_groups):
            coefs[:, j, :] = 1
            coefs[:, j, :] *= problem.card_sizes
            problem.G.append(coefs.flatten())
            problem.h.append(problem.recommendation_groups[j].size)
            coefs[:, j, :] = 0


class RecommendationGroupAdjacencyConstraint(CustomConstraint):
    """
    Does not allow equal adjacent recommendation groups.
    """
    hint = "Restrict adjacent recommenders"

    def apply(self, problem):
        if problem.n_recommendation_groups < 2:
            # cannot satisfy adjacency constraint with only one recommendation group type
            return
        coefs = np.zeros(problem.dimensions, dtype=np.int32)
        for i in xrange(problem.n_blocks - 1):
            for j in xrange(problem.n_recommendation_groups):
                coefs[i, j, :] = 1
                coefs[i + 1, j, :] = 1
                problem.G.append(coefs.flatten())
                problem.h.append(1)
                coefs[i, j, :] = 0
                coefs[i + 1, j, :] = 0


class CardAdjacencyConstraint(CustomConstraint):
    """
    Does not allow equal adjacent card types.
    """
    hint = "Restrict adjacent card types"

    def get_last_card_type(self, problem):
        if problem.previous_page is not None:
            last_card_type = problem.previous_page.blocks[-1].card_type
            return last_card_type

    def apply(self, problem):
        if problem.n_cards < 2:
            # cannot satisfy adjacency constraint with only one card type
            return
        coefs = np.zeros(problem.dimensions, dtype=np.int32)
        # we can have different cards allowed sharing the same basic type
        # (and differing in app quantity). such cards are restricted too:

        card_types = defaultdict(list)
        for i, card in enumerate(problem.cards):
            card_types[card.card_type].append(i)

        # if this page is not the first one, adjacency constraint
        # should also consider previous page
        last_card_type = self.get_last_card_type(problem)
        if last_card_type is not None and last_card_type in card_types:
            idx = card_types[last_card_type]
            coefs[0, :, idx] = 1
            problem.A.append(coefs.flatten())
            problem.b.append(0)
            coefs[0, :, idx] = 0

        for i in xrange(problem.n_blocks - 1):
            for card_type, idx in card_types.iteritems():
                coefs[i, :, idx] = 1
                coefs[i + 1, :, idx] = 1
                problem.G.append(coefs.flatten())
                problem.h.append(1)
                coefs[i, :, idx] = 0
                coefs[i + 1, :, idx] = 0


class BalancedRecommenderConstraint(CustomConstraint):
    """
    Ensures feed isn't dominated by small subset of recommenders.
    Two different scenarios are possible here:

     * there are more blocks than recommenders: therefore
       we can set a lower bound for each, which will be equal
       to min(recommender_capacity / min_card_size, n_blocks / n_recommenders)
     * there are more recommenders than blocks: lower bound
       is therefore impossible (we can't fit all recommendations in
       smaller amount of blocks). An upper bound is introduced instead:
       no recommender can occupy more then one block.

    Notice we operate recommenders here (not recommendation groups).
    """
    hint = "Try to balance recommenders"

    def apply_lower_bound(self, coefs, problem, idx, n_places, n_groups, capacity, min_card_size):
        """
        :param coefs: binary coefficients for integer problem
        :param problem: problem instanse
        :param idx: index (or array of indices) of recommendation groups to apply lower bound to
        :param n_places: number of blocks to occupy
        :param n_groups: number of recommendation groups available
        :param capacity: overall capacity of all groups
        :param min_card_size: minimal card size
        """
        # exclude penalized recommenders (those with negative weights)
        # NOTE: this is very non-obvious, but the whole strategy module
        # needs refactoring anyway
        group_weights = problem.group_weights[idx]
        valid_idx = idx[group_weights >= 0]
        if len(valid_idx) != len(idx):
            logger.debug(
                "Not applying lower bound to some recommendations groups "
                "because they are penalized (have negative group_weights). "
                "Requested groups %s, leaving %s",
                idx, valid_idx
            )
            if len(valid_idx) == 0:
                logger.debug("No recommendation groups to apply lower bound to; exiting")
                return

        coefs[:, valid_idx, :] = 1
        bound = min(n_places / n_groups, capacity / min_card_size)
        # NOTE: lower bound means multiplying by -1 left and right sides
        problem.G.append(coefs.flatten() * -1)
        problem.h.append(bound * -1)
        coefs[:, valid_idx, :] = 0
        return bound

    def apply_upper_bound(self, coefs, problem, idx):
        coefs[:, idx, :] = 1
        problem.G.append(coefs.flatten())
        problem.h.append(1)
        coefs[:, idx, :] = 0
        return 1

    def apply(self, problem):
        # group recommendation groups by recommenders
        recommender_indices = defaultdict(list)
        recommender_group_capacities = defaultdict(list)

        for i, group in enumerate(problem.recommendation_groups):
            recommender_indices[group.recommender].append(i)
            recommender_group_capacities[group.recommender].append(group.size)

        recommender_indices = {k: np.array(v) for k, v in recommender_indices.iteritems()}
        n_recommenders = len(recommender_indices)

        # apply constraint to recommenders
        coefs = np.zeros(problem.dimensions, dtype=np.int32)

        # since this constraint may be tricky to satisfy, we'll collect
        # debug information in case of a possible failure
        debug = {}

        for recommender, group_idx in recommender_indices.iteritems():
            # determine min_card_size for this recommender
            min_card_size = min(card.size for card in problem.cards
                                if problem.recommendation_groups[group_idx[0]].content_type in card.supported_content_types)

            if max(recommender_group_capacities[recommender]) < min_card_size:
                # no group sufficient enough to fill even one card. cannot balance.
                continue

            if problem.n_blocks > n_recommenders:
                recommender_bound = self.apply_lower_bound(
                    coefs, problem, group_idx,
                    n_places=problem.n_blocks,
                    n_groups=n_recommenders,
                    capacity=sum(recommender_group_capacities[recommender]),
                    min_card_size=min_card_size
                )
                recommender_bound_type = 'lower'
            else:
                recommender_bound = self.apply_upper_bound(coefs, problem, group_idx)
                recommender_bound_type = 'upper'

            debug[recommender.experiment_name] = {
                'capacity': sum(recommender_group_capacities[recommender]),
                'bound_type': recommender_bound_type,
                'bound': recommender_bound,
                'groups': []
            }

            # then apply constraint to recommendation groups within recommender
            n_groups = len(group_idx)
            for j in group_idx:
                if recommender_bound_type == 'upper':
                    # recommender cannot fill more then one block.
                    # nothing to do here (sum of its group coefficients is already
                    # bounded by 1 by previous `apply_upper_bound` call)
                    group_bound = recommender_bound
                    group_bound_type = 'upper'
                elif recommender_bound > len(group_idx):
                    # recommender has to fill some minimal number of blocks AND it has
                    # less groups than number of blocks. we should balance its groups
                    # so that each takes approximately fair share of blocks
                    group_bound = self.apply_lower_bound(
                        coefs, problem, np.array([j]),
                        n_places=recommender_bound,
                        n_groups=n_groups,
                        capacity=problem.recommendation_groups[j].size,
                        min_card_size=min_card_size
                    )
                    group_bound_type = 'lower'
                else:
                    # the ambiguous case: recommender is lower-bounded, but has
                    # too many groups to demand fair share balancing. we cannot
                    # enforce upper bound here (see ADVISOR-1203) because we don't
                    # know an upper bound for recommender overall. the only solution
                    # here is skipping the constraint for groups altogether
                    break
                debug[recommender.experiment_name]['groups'].append({
                    'capacity': problem.recommendation_groups[j].size,
                    'bound_type': group_bound_type,
                    'bound': group_bound
                })
        return debug


def get_promo_groups_indexes(problem):
    return np.array([
        i for i, group in enumerate(problem.recommendation_groups)
        if group.content_type == 'promo'
    ], dtype=np.int32)


class ContentTypeMappingConstraint(MandatoryConstraint):
    """
    Put recommendation groups only to cards with supported content types
    """

    def apply(self, problem):
        group_types = {group.content_type for group in problem.recommendation_groups}

        for type in group_types:
            coefs = np.zeros(problem.dimensions, dtype=np.int32)

            type_groups_idx = np.array([
                i for i, group in enumerate(problem.recommendation_groups)
                if group.content_type == type
            ], dtype=np.int32)

            restricted_cards_idx = np.array([
                i for i, card in enumerate(problem.cards)
                if type not in card.supported_content_types
            ], dtype=np.int32)

            # all groups fit all cards
            if len(restricted_cards_idx) == 0:
                continue

            assert len(restricted_cards_idx) < len(problem.cards), 'No cards support content type "%s"' % type

            for group_idx in type_groups_idx:
                coefs[:, group_idx, restricted_cards_idx] = 1

            problem.A.append(coefs.flatten())
            problem.b.append(0)


class PromoCardNotFirstConstraint(CustomConstraint):
    """
    This constraint is used in feed/gamehub recommendations.
    First block should not be promo.
    """
    hint = "First card should not be promo"

    def apply(self, problem):
        coefs = np.zeros(problem.dimensions, dtype=np.int32)
        promo_idx = get_promo_groups_indexes(problem)
        coefs[0, promo_idx, :] = 1
        problem.A.append(coefs.flatten())
        problem.b.append(0)


class Strategy(object):
    """
    Feed strategy solves a problem which generally looks like this:
    there's a 3-dimensional cube of n_blocks x n_recommendation_groups x n_cards cells.
    Each cell is binary, and an active cell with coordinates (i, j, k) corresponds to the
    fact that ith feed block will be filled with recommendations from jth group and
    kth card type. A set of additional constraints on cell activations includes some of
    the following:

     * only one recommender/card per position: i.e., sum(cube[i, :, :]) must be equal to 1
     * capacity constraint per recommendation groups:
       cube[:, j, 0] * card_size_0 + ... + cube[:, j, n] * card_size_n must be less than
       or equal to recommendation group size
     * recommender adjacency constraints ("round-robin strategy"):
       for any i, k, cube[i, j, k] must not be equal to cube[i, j + 1, k].
     * card adjacency constraints ("non-repeating card strategy"): analogous to the previous one.
     * recommendation group balance constraint: to prevent a situation when two or three recommenders
       dominate the entire feed, sum(cube[:, j, :]) must be around n_blocks / n_recommendation_groups
       (if jth group capacity allows this).
     * specific card constraints: if kth card cannot be promo, sum(cube[:, promo_group_idx, k])
       must be equal to 0.

    ...and so on. The problem is solved via integer programming, so additional constraints can be
    added as long as they can be represented by inequality or equality expressions.
    """

    def __init__(self, constraints, card_configs, card_weights, randomization):
        """
        :param constraints: collection of instances of Constraint class
        :param card_configs: collection of (card config class, card config params) tuples
        :param card_weights: array of weights/objective coefficient multipliers for integer program
                             (must have the same length as `card_configs`)
        :param randomization: a single float, controls the magnitude of random scores
        """
        assert len(card_configs) == len(card_weights)
        mandatory_constraints = [constraint() for constraint in get_all_subclasses(MandatoryConstraint)]
        self.constraints = mandatory_constraints + constraints
        self.card_configs = card_configs
        self.card_weights = card_weights
        self.randomization = randomization

    def init_cards(self, user):
        """
        Merges user card configs with mongo configs in configuration database
        :param user: an instance of `jafar.models.user.User`

        :returns: a pair of (`Card` objects, corresponding weights)
        """
        cards = []
        weights = []
        for weight, (config_class, params) in zip(self.card_weights, self.card_configs):
            client_card_config = user.supported_card_types.get(config_class.card_type)
            if client_card_config:
                card = config_class(client_count_limit=client_card_config['count'], **params).get_card()
                cards.append(card)
                weights.append(weight)
        if not cards:
            raise FeedException("Strategy doesn't support any cards allowed by user")
        return cards, np.array(weights)

    def debug_failed_problem(self, feed, block_count, recommendation_groups, cards, card_weights, group_weights):
        """
        Tries to apply constraints one by one and find the one which is
        unsatisfiable.
        """
        for step in xrange(1, len(self.constraints) + 1):
            problem = Problem(block_count, cards, recommendation_groups, card_weights, group_weights, self.randomization)
            for constraint in self.constraints[:step]:
                debug_info = constraint.apply(problem)
            try:
                problem.solve()
            except FeedException:
                msg = "Failed to find a solution for an integer programming problem: " \
                      "constraint {} cannot be satisfied".format(self.constraints[step - 1].__class__.__name__)
                raise FeedException(msg, extra=debug_info)

    def solve_integer_problem(self, feed, recommendation_groups, previous_page, cards, card_weights, group_weights):
        n_recommendations = sum(group.size for group in recommendation_groups)
        block_count = self.truncate_block_count(feed.block_count, cards, n_recommendations)

        problem = Problem(
            block_count, cards, recommendation_groups,
            card_weights, group_weights,
            self.randomization, previous_page
        )
        for constraint in self.constraints:
            constraint.apply(problem)
        try:
            solution_coefs = problem.solve()
        except FeedException:
            self.debug_failed_problem(feed, block_count, recommendation_groups, cards, card_weights, group_weights)
            raise

        # current solution is a vector of coefficients: convert it to dict (group, [cards])
        solution_coefs = solution_coefs.reshape(problem.dimensions)

        solution = defaultdict(lambda: (list(), list()))
        for block_index in xrange(problem.n_blocks):
            group_idx, card_idx = solution_coefs[block_index].nonzero()
            solution[recommendation_groups[group_idx[0]]][0].append(cards[card_idx[0]])
            solution[recommendation_groups[group_idx[0]]][1].append(block_index)  # we'll rearrange them by block number
        return solution

    def truncate_block_count(self, n_blocks, cards, n_recommendations):
        """
        Depending on available recommenders, it might not be possible
        to fill the entire amount of blocks. Rather than throwing an
        exception, we'll try to truncate the number of blocks required.

        Worst-case scenario will be one smallest card
        But it's possible that it will not fit CardAdjacencyConstraint

        """
        smallest_card = min(cards, key=lambda card: card.size)  # TODO: differ gifts/apps cards
        maximum_possible_n_blocks = n_recommendations / smallest_card.size
        if maximum_possible_n_blocks < 1:
            logger.info('Not enough recommendations to fill even one block')
            raise NotFound("Empty recommendations")

        if maximum_possible_n_blocks < n_blocks:
            logger.info("Truncating block count from %s to %s", n_blocks, maximum_possible_n_blocks)
            return maximum_possible_n_blocks
        return n_blocks

    def get_blocks(self, feed, previous_page=None):
        # step 0: initialize cards and their weights
        cards, card_weights = self.init_cards(feed.user)

        # step 1: get recommendation groups from all recommenders
        # at the same time, discard groups which has recently been shown to user
        recommendation_groups = []
        for recommender in feed.recommenders:
            try:
                for group in recommender.get_recommendation_groups(feed, cards):
                    if group.explanation not in feed.meta.explanation_filter:
                        recommendation_groups.append(group)
            except RecommenderNotReady:
                logger.error('Recommender %s is not ready!', recommender.experiment_name)
                pass

        if not recommendation_groups:
            raise NotFound('Empty recommendations')

        # step 2: filter out recommendations from previous page (if present)
        for group in recommendation_groups:
            group.filter(feed.meta.item_filter)

        # step 3: drop inter-group duplicates. before that, sort groups according
        # to size, for smaller groups to get the chance to keep their recommendations.
        duplicate_item_filter = set()
        for group in sorted(recommendation_groups, key=lambda group: group.size):
            group.filter(duplicate_item_filter)
            group.update_filter(duplicate_item_filter)

        # leave only not empty recommendation groups which have corresponding supported types
        output_types = {type for card in cards for type in card.supported_content_types}

        filtered_groups = []
        for group in recommendation_groups:
            if group.content_type in output_types and group.size > 0:
                filtered_groups.append(group)

        for group in filtered_groups:
            logger.debug(
                "Got %s recommendations by %s in a group with an explanation %s",
                group.size, group.recommender.__class__.__name__, group.explanation
            )

        # step 4: now when all recommendations in groups are valid, we can proceed
        # to solving an integer programming problem
        group_weights = np.array([group.weight for group in filtered_groups])
        solution = self.solve_integer_problem(
            feed, filtered_groups, previous_page, cards, card_weights, group_weights
        )

        blocks = []
        block_indices = []
        for group, (cards, index) in solution.iteritems():
            blocks.extend(group.get_blocks(cards))
            block_indices.extend(index)
        blocks = list(np.array(blocks)[np.argsort(block_indices)])
        return blocks
