from pybloom_live import ScalableBloomFilter


class BloomItemFilter(object):

    def __init__(self, items=None, bloom_filter=None):
        if bloom_filter:
            self.content = bloom_filter
        else:
            self.content = self._get_bloom_filter(items)

    @classmethod
    def _get_bloom_filter(cls, items=None):
        if items is None:
            items = []
        bloom_filter = ScalableBloomFilter(
            mode=ScalableBloomFilter.SMALL_SET_GROWTH
        )
        for item in items:
            bloom_filter.add(item)
        return bloom_filter

    def add(self, item):
        self.content.add(item)

    def update(self, items):
        for item in items:
            self.add(item)

    def __contains__(self, item):
        return item in self.content


class MultipleBloomItemFilter(BloomItemFilter):
    """
    Maintains specified number of bloom filters
    (one per feed page, for example), discarding older
    ones as the content size grows.
    """

    def __init__(self, num_filters, items=None, bloom_filters=None):
        self.num_filters = num_filters
        self.append = False
        if bloom_filters:
            self.content = bloom_filters
        elif num_filters:
            self.content = [self._get_bloom_filter(items)]
        else:
            self.content = []

    def add(self, item):
        if not self.num_filters:
            return
        if not self.append:
            if len(self.content) >= self.num_filters:
                self.content.pop(0)
            self.content.append(self._get_bloom_filter())
            self.append = True
        self.content[-1].add(item)

    def __contains__(self, item):
        return any(item in bloom for bloom in self.content)
