# coding=utf-8
import logging
import math
import os

from PIL import Image, ImageFont, ImageDraw

from settings import STATIC_DIR

logger = logging.getLogger(__name__)

SIZE_TITLE_DP = 22
SIZE_SUBTITLE_DP = 13
IMG_WIDTH_DP = 332
IMG_HEIGHT_DP = 130
CROPPED_WIDTH_DP = math.ceil(IMG_WIDTH_DP * .55)

MARGIN_START_DP = 16
TEXT_GAP_DP = 6
MARGIN_TOP_ONE_TITLE_DP = 54
MARGIN_TOP_TWO_TITLES_DP = 44

TITLE_LIMIT = 16
SUBTITLE_LIMIT = 24

MODE = 'RGBA'

# The biggest dpi, is used for generating covers for old clients
XXXHDPI_DENSITY = 640

class Title(object):
    def __init__(self, text, font, color):
        self.text = text
        self.font = font
        self.color = color


class ImageComposer(object):
    title = None
    subtitle = None
    gradient_color = None
    full_size = None

    def __init__(self, picture_name, background, dpi):
        self.picture_name = picture_name
        self.background_color = background
        self.dpi = dpi

    def _to_px_tuple(self, x_dp, y_dp):
        return (
            self.dp_to_px(x_dp, self.dpi),
            self.dp_to_px(y_dp, self.dpi)
        )

    def _draw_text(self, image):
        if self.subtitle:
            margin_start = self.dp_to_px(MARGIN_START_DP, self.dpi)
            text_gap = self.dp_to_px(TEXT_GAP_DP, self.dpi)
            title_y = self.dp_to_px(MARGIN_TOP_TWO_TITLES_DP, self.dpi)
            title_font_height = 0
            if self.title:
                title_font_height = self.title.font.font.height
            subtitle_y = text_gap + title_y + title_font_height

            if self.title:
                self.draw_title(image, self.title, (margin_start, title_y))
            self.draw_title(image, self.subtitle, (margin_start, subtitle_y))
        elif self.title:
            position = self._to_px_tuple(MARGIN_START_DP, MARGIN_TOP_TWO_TITLES_DP)
            self.draw_title(image, self.title, position)
        return image

    def _adjust_image_resize(self, image, target_width):
        if image.width == image.height:
            resize_width = target_width
            resize_height = target_width
        else:
            width_percent = target_width / float(image.width)
            height_size = int(float(image.height) * float(width_percent))
            resize_width = target_width
            resize_height = height_size
        return image.resize((resize_width, resize_height), Image.ANTIALIAS)

    def _adjust_image_crop(self, image, target_width, background_height, background_width):
        """
        If Image is square, then crop center the image, to discard equal parts of the image at top and bottom
        For example, if input image is(it doesn't look like a square, but it is):

        11111111
        22222222
        33333333
        44444444
        55555555
        66666666
        77777777
        88888888

        And the target height is 4, then result image will be:

        33333333
        44444444
        55555555
        66666666


        If image is panorama, then crop center image image by width and height, to leave only the center of image
        For example, if input image is(height is 5, width is 16):

        1234567117654321
        1234567227654321
        1234567337654321
        1234567447654321
        1234567557654321

        And the target height is 3, and target width is 4, result will be:

        7227
        7337
        7447

        :return: cropped image
        """
        height_delta = (image.height - background_height) / 2
        width_delta = 0
        width_crop_end = target_width
        if image.width != image.height and image.width > target_width and not self.full_size:
            width_delta = (background_width - target_width) / 2
            width_crop_end = image.width - width_delta
        if height_delta == 0 and width_delta == 0:
            return image
        return image.crop(
            (width_delta, math.ceil(height_delta), width_crop_end, image.height - math.floor(height_delta)))

    def _adjust_image(self, image):
        background_width = self.dp_to_px(IMG_WIDTH_DP, self.dpi)
        width = background_width if self.full_size else self.dp_to_px(CROPPED_WIDTH_DP, self.dpi)
        background_height = self.dp_to_px(IMG_HEIGHT_DP, self.dpi)
        image = self._adjust_image_resize(image, width)
        image = self._adjust_image_crop(image, width, background_height, background_width)
        return image

    def _add_background(self, image):
        background_size = self._to_px_tuple(IMG_WIDTH_DP, IMG_HEIGHT_DP)
        background = Image.new(MODE, background_size, self.hex_to_rgb(self.background_color))

        width_start = background.width - image.width
        width_end = width_start + image.width
        height_start = 0
        height_end = image.height

        background.paste(image, (
            width_start,
            height_start,
            width_end,
            height_end
        ))

        return background

    def _add_gradient(self, image):
        if self.gradient_color:
            gradient_size = self._to_px_tuple(IMG_WIDTH_DP, IMG_HEIGHT_DP)
            gradient = self.create_gradient(self.hex_to_rgb(self.gradient_color), gradient_size)
            image.alpha_composite(gradient)
        return image

    def set_title(self, text, color):
        font = self.load_font('YS-Text-Bold.ttf', SIZE_TITLE_DP, self.dpi)
        text = self.ellipsize(text, TITLE_LIMIT)
        self.title = Title(text, font, self.hex_to_rgb(color))

    def set_subtitle(self, text, color):
        font = self.load_font('YS-Text-Regular.ttf', SIZE_SUBTITLE_DP, self.dpi)
        text = self.ellipsize(text, SUBTITLE_LIMIT)
        self.subtitle = Title(text, font, self.hex_to_rgb(color))

    def add_gradient(self, color=None):
        self.gradient_color = color or self.background_color

    def set_full_size(self, full_size):
        self.full_size = full_size

    def compose(self):
        with Image.open(self.picture_name) as image:
            if image.mode != MODE:
                image = image.convert(MODE)
            image = self._adjust_image(image)
            image = self._add_background(image)
            image = self._add_gradient(image)
            image = self._draw_text(image)
            return image

    @staticmethod
    def load_font(name, size, density):
        return ImageFont.truetype(
            os.path.join(STATIC_DIR, 'fonts', name),
            ImageComposer.dp_to_px(size, density),
            encoding='UTF-8'
        )

    @staticmethod
    def draw_title(image, title, position):
        ImageDraw.Draw(image).text(
            position,
            title.text,
            title.color,
            title.font
        )
        return image

    @staticmethod
    def hex_to_rgb(hex_color):
        """
        Converts a hex color string to an RGB or an RGBA tuple.
        If a parameter is not a string, return itself.
        """
        if type(hex_color) is str:
            raw_hex = hex_color.lstrip('#')

            if len(raw_hex) == 8:
                mask = (2, 4, 6, 0)
            elif len(raw_hex) == 6:
                mask = (0, 2, 4)
            else:
                raise ValueError('Unknown color format %s' % hex_color)

            rgb_color = tuple(int(raw_hex[i:i + 2], 16) for i in mask)
        elif type(hex_color) is tuple:
            rgb_color = hex_color
        else:
            raise ValueError('Unknown color format %s' % hex_color)

        for c in rgb_color:
            if c < 0 or c > 255:
                raise ValueError('Incorrect value %d: %s results to %s' % (c, hex_color, rgb_color))

        return rgb_color

    @staticmethod
    def dp_to_px(dp, density):
        """
        Converts density independent points to pixels.
        Taken from advisor/common/screen_properties
        """
        px = int(dp) * int(density) / 160
        return int(px)

    @staticmethod
    def create_gradient(color, size):
        gradient = Image.open(os.path.join(STATIC_DIR, 'images', 'card_gradient.png'))
        gradient = gradient.convert(MODE)
        gradient = gradient.resize(size)
        pixels = gradient.load()
        for y in range(gradient.size[1]):
            for x in range(gradient.size[0]):
                old_r, old_g, old_b, old_a = pixels[x, y]
                if len(color) == 3:
                    new_r, new_g, new_b = color
                elif len(color) == 4:
                    new_r, new_g, new_b, _ = color
                else:
                    raise ValueError('Cannot unpack color %s' % color)
                if (old_r, old_g, old_b) == (0, 0, 0):
                    pixels[x, y] = (new_r, new_g, new_b, old_a)
        return gradient

    @staticmethod
    def ellipsize(text, limit):
        if len(text) > limit + 3:
            text = text[:limit] + '...'
        return text
