import abc
import random

from jafar.web.exceptions import BadRequest

MAX_WIDGET_ITEMS = 6


class Card(object):
    """
    Instance of card, generated using card config
    """
    content_type = 'apps'

    def __init__(self, reserved_count, card_type, min_count=1, max_count=None,
                 supported_content_types=('apps',)):
        self.min_count = min_count
        self.max_count = max_count
        self.reserved_count = reserved_count
        self.card_type = card_type
        self.rotation_interval = None
        self.supported_content_types = supported_content_types

    @property
    def size(self):
        return self.min_count + self.reserved_count

    def __repr__(self):
        return '<%s card size: %d>' % (self.card_type, self.size)


class GiftCard(Card):
    """
    Special card type for YPhone users gifts
    """
    content_type = 'gifts'

    def __init__(self, reserved_count, card_type,
                 min_count=None, max_count=None, supported_content_types=('gifts',)):
        super(GiftCard, self).__init__(reserved_count, card_type,
                                       min_count, max_count, supported_content_types)
        self.reserved_count = 0


class CardConfig(object):
    """
    Card config is the special object constructed from information
    about the client possibilities and backend configuration.

    Describes the way how the Card instances should be created

    """
    __metaclass__ = abc.ABCMeta

    count_is_required = True
    card_type = None

    def __init__(self, min_count=1, max_count=None, client_count_limit=None):
        if self.count_is_required and client_count_limit is None:
            raise CardConfigError(card_type=self.card_type, reason='count from client is required')
        count = client_count_limit or min_count or max_count
        self.min_count = count
        self.max_count = count

    def get_card(self):
        return Card(
            min_count=self.min_count,
            max_count=self.max_count,
            reserved_count=max(self.min_count, 1),
            card_type=self.card_type,
            supported_content_types=('apps', 'promo')
        )


class RandomSingleAppCardConfig(CardConfig):
    card_type = 'Singleapp_card_Expandable'
    count_is_required = False

    def get_card(self):
        """
        We have a random count range [min_count, max_count], including both end points.
        """
        count = random.randint(self.min_count, self.max_count)
        return Card(
            min_count=count,
            max_count=count,
            reserved_count=max(count - 1, 1),
            card_type=self.card_type,
            supported_content_types=('apps',)
        )


class SingleAppCardConfig(CardConfig):
    card_type = 'Singleapp_card_Expandable'
    count_is_required = False

    def __init__(self, count=1, **kwargs):
        super(SingleAppCardConfig, self).__init__(**kwargs)
        self.count = count

    def get_card(self):
        """
        SingleAppCard Card with count defined in mongo config
        """
        return Card(
            min_count=self.count,
            max_count=self.count,
            reserved_count=max(self.count - 1, 1),
            card_type=self.card_type,
            supported_content_types=('apps',)
        )


class MultiappsRichCardConfig(CardConfig):
    card_type = 'Multiapps_card_rich'


class MultiappsCardConfig(CardConfig):
    card_type = 'Multiapps_card'


class MultiappsMultirowsCardConfig(CardConfig):
    card_type = 'Multiapps_card_multirows'


class RotatableCardConfig(CardConfig):
    """ Rotatable cards used in launcher categories and widget. Thay have rotattion interval - parameter  """

    def __init__(self, rotation_interval=0, **kwargs):
        super(RotatableCardConfig, self).__init__(**kwargs)
        self.rotation_interval = rotation_interval

    def get_card(self):
        card = super(RotatableCardConfig, self).get_card()
        card.rotation_interval = self.rotation_interval
        return card


class MultiappsWidgetCardConfig(RotatableCardConfig):
    card_type = 'Multiapps_widget'


class ExpandableButtonCardConfig(RotatableCardConfig):
    card_type = 'Expandable_button'


class ScrollableCardConfig(RotatableCardConfig):
    card_type = 'Scrollable'

    def __init__(self, min_count=1, max_count=None, client_count_limit=None, **kwargs):
        super(ScrollableCardConfig, self).__init__(min_count=min_count, max_count=max_count,
                                                   client_count_limit=client_count_limit, **kwargs)
        self.min_count = min(min_count, client_count_limit)
        self.max_count = min(max_count, client_count_limit) or max_count or client_count_limit

    def get_card(self):
        card = super(ScrollableCardConfig, self).get_card()
        card.reserved_count = 3
        card.supported_content_types = ('apps',)
        return card


class SetupWizardCardConfig(ScrollableCardConfig):
    card_type = 'Setup_wizard_card'

    def get_card(self):
        card = super(SetupWizardCardConfig, self).get_card()
        card.reserved_count = 0
        return card


class ScrollableExpandableCardConfig(ScrollableCardConfig):
    card_type = 'Scrollable_Expandable'


class CardConfigError(BadRequest):
    def __init__(self, card_type, reason):
        super(CardConfigError, self).__init__(details={"errors": [
            "Bad {card_type} config: {reason}".format(card_type=card_type, reason=reason)
        ]})


class ZenSingleappCardConfig(SingleAppCardConfig):
    card_type = 'Zen_Singleapp_card'

    def get_card(self):
        card = super(ZenSingleappCardConfig, self).get_card()
        card.reserved_count = 0
        return card


class CustomUICardConfig(CardConfig):
    """
    Special Custom UI card https://st.yandex-team.ru/ADVISOR-1673.
    Item count is loaded from rec_view_config, reserve is empty.
    """
    card_type = 'CustomUI_card'

    def get_card(self):
        return Card(
            min_count=self.min_count,
            max_count=self.max_count,
            reserved_count=0,
            card_type=self.card_type,
            supported_content_types=('apps', 'promo')
        )


class GiftsCardConfig(CardConfig):
    """
    Special card config type for YPhone users gifts
    """

    card_type = 'Bonus_universal_card'

    def get_card(self):
        return GiftCard(
            min_count=self.min_count,
            max_count=self.max_count,
            card_type=self.card_type,
            reserved_count=0
        )
