import logging
from datetime import datetime
from hashlib import sha256
from uuid import UUID

import colorwiz.tool as colorwiz
from mongoengine import (EmbeddedDocument, EmbeddedDocumentField, StringField, ListField, DictField, URLField,
                         ValidationError, IntField, DateTimeField, BooleanField)

from jafar import db
from jafar.admin.mds import LauncherMdsClient

logger = logging.getLogger(__name__)

launcher_mds_client = LauncherMdsClient()

update_feeds_logger = logging.getLogger('update_feeds')

DISK_ID = 'disk'
MONEY_ID = 'money'
TAXI_ID = 'taxi'
PLUS_ID = 'plus'

ALL_GIFTS = (DISK_ID, MONEY_ID, TAXI_ID, PLUS_ID)


class Region(db.EmbeddedDocument):
    code = db.IntField(required=True)
    description = db.StringField(required=True)

    meta = {'strict': False}

    def __unicode__(self):
        return self.description


class LangSpecificValues(EmbeddedDocument):
    lang = StringField()
    title = StringField()
    description = StringField()

    def __unicode__(self):
        return u'%s: %s - %s' % (self.lang, self.title, self.description)


class YandexDistributedApp(db.Document):
    package_name = StringField(required=True, unique=True)
    texts = ListField(field=EmbeddedDocumentField(LangSpecificValues))
    # Now offer_id field is not connected to any model in mongo
    # But we have some logic in the advisor based on it, so we should not delete it.
    offer_id = StringField(required=False)
    show_in_feed = BooleanField(required=False, default=False)
    tag = StringField(required=False, choices=('popular', 'mobile', 'bank', 'yandex', 'dont_show'))
    tag_extra = DictField(required=False)
    content_rating = StringField(choices=('3+', '7+', '12+', '16+', '18+'))
    download_url = URLField()
    icon = StringField(required=False)
    rank = IntField(required=False, default=0)
    regions = db.ListField(db.EmbeddedDocumentField(Region), required=False)
    mark_as_sponsored = BooleanField(required=False, default=False)

    def clean(self):
        if self.tag == 'mobile' and (
                self.tag_extra is None or
                self.tag_extra.get('mcc', None) is None or
                self.tag_extra.get('mnc', None) is None):
            raise ValidationError(
                'Application with "mobile" tag should have "mcc" and "mnc" fields in tag_extra'
            )

        if self.tag == 'bank' and (self.tag_extra is None or not self.tag_extra.get('uniq_name_part', None)):
            raise ValidationError(
                'Application with "bank" tag should have non empty "uniq_name_part" field in tag_extra'
            )

    def get_text(self, language):
        for text in self.texts:
            if text.lang == language:
                return text

    meta = {
        'db_alias': 'advisor',
        'auto_create_index': False,
        'indexes': ['tag', 'package_name'],
        'strict': False
    }


class Gifts(db.Document):
    id = StringField(primary_key=True)
    package_name = StringField(required=True, regex='^gift:[a-zA-Z0-9_.]+')
    preview_order = IntField(required=False, db_field='order')
    suw_preview_image = StringField(required=True)
    updated_at = DateTimeField(required=True, default=datetime.utcnow)

    meta = {
        'db_alias': 'advisor',
        'strict': False,
    }

    def __unicode__(self):
        return self.id

    def save(self, *args, **kwargs):
        self.updated_at = datetime.utcnow()
        super(Gifts, self).save(*args, **kwargs)


class BonusCardText(EmbeddedDocument):
    lang = StringField(required=True)
    app_name = StringField(required=True)
    title = StringField(required=True)
    description = StringField(required=True)
    disclaimer = StringField(required=False, default=str)


class BonusCardResources(db.Document):
    package_name = StringField(primary_key=True)
    icon = StringField(required=True)
    banner_image = StringField(required=True)
    url_template = StringField(required=True)
    button_text_key = StringField(required=True)
    updated_at = DateTimeField(required=True, default=datetime.utcnow)
    texts = ListField(field=EmbeddedDocumentField(BonusCardText))
    mark_as_sponsored = BooleanField(required=False, default=False)
    content_rating = StringField(required=False, default=str)

    icon_colorwiz = DictField(required=False)
    banner_image_colorwiz = DictField(required=False)

    meta = {
        'db_alias': 'advisor',
        'strict': False,
    }

    @staticmethod
    def calculate_colorwiz(image_key):
        image_url = launcher_mds_client.read_url(image_key)
        return colorwiz.process_url(image_url)

    def __unicode__(self):
        return self.id

    def get_text(self, language):
        for text in self.texts:
            if text.lang == language:
                return text

    def save(self, *args, **kwargs):
        if not self.banner_image_colorwiz:
            self.banner_image_colorwiz = self.calculate_colorwiz(self.banner_image)
        if not self.icon_colorwiz:
            self.icon_colorwiz = self.calculate_colorwiz(self.icon)
        self.updated_at = datetime.utcnow()
        super(BonusCardResources, self).save(*args, **kwargs)


class GivenGift(db.EmbeddedDocument):
    gift = db.ReferenceField(Gifts, required=True, reverse_delete_rule=db.DO_NOTHING)
    activated = db.BooleanField(required=False, default=False)
    promocode = db.StringField(required=False)
    activated_at = db.DateTimeField(required=False)

    def __unicode__(self):
        return u'GivenGift: {}'.format(self.gift.pk)


class GiftSetError(Exception):
    pass


class GiftSet(db.Document):
    passport_uid = db.LongField(primary_key=True)
    gifts = db.ListField(db.EmbeddedDocumentField(GivenGift))
    created_at = db.DateTimeField(required=True, default=datetime.utcnow)

    def __unicode__(self):
        return u'GiftSet: {}'.format(self.passport_uid)

    def clean(self):
        unique_gift_ids = set()
        gifts = []
        # removing duplicate gifts or disk gift
        for given_gift in self.gifts:
            gift_id = given_gift.gift.pk
            if gift_id in unique_gift_ids or gift_id == DISK_ID:
                continue
            gifts.append(given_gift)
            unique_gift_ids.add(gift_id)
        self.gifts = gifts

    def get_given_gift_by_id(self, gift_id):
        for given_gift in self.gifts:
            if given_gift.gift.pk == gift_id:
                return given_gift
        else:
            raise GiftSetError("No GivenGift with id = {gift_id} found in {gift_set}".format(
                gift_id=gift_id, gift_set=self))

    def activate_gift(self, gift_id):
        given_gift = self.get_given_gift_by_id(gift_id)
        if not given_gift.activated:
            given_gift.activated = True
            given_gift.activated_at = datetime.utcnow()

    meta = {
        'db_alias': 'advisor',
        'strict': False,
    }


def luhn_checksum(data):
    digits = map(int, data)
    odd_digits = digits[-1::-2]
    even_digits = digits[-2::-2]
    checksum = 0
    checksum += sum(odd_digits)
    even_digits = ''.join([str(d * 2) for d in even_digits])
    checksum += sum(map(int, even_digits))
    return checksum % 10


class ImeiField(db.StringField):
    def __init__(self, *args, **kwargs):
        kwargs['min_length'] = kwargs['max_length'] = 15
        kwargs['regex'] = r'^[0-9]{15}$'
        super(ImeiField, self).__init__(*args, **kwargs)

    def validate(self, value):
        super(ImeiField, self).validate(value)
        if luhn_checksum(value) != 0:
            self.error('IMEI Checksum error')


class Phone(db.Document):
    id = db.StringField(primary_key=True)
    device_id = db.UUIDField(required=False)
    activated = db.BooleanField(required=False, default=False)
    first_activation_datetime = db.DateTimeField(required=False)
    last_activation_datetime = db.DateTimeField(required=False)
    # gifts info
    gift_set = db.ReferenceField(GiftSet, reverse_delete_rule=db.NULLIFY)
    # each taxi promocode is strictly tied to specific Phone object
    taxi_promocode = db.StringField(required=False)
    # Some useful info about phones
    imei = ImeiField(required=False)
    imei2 = ImeiField(required=False)
    serial_number = db.StringField(required=False)
    wifi_mac = db.StringField(required=False)
    bluetooth_mac = db.StringField(required=False)
    batch = db.StringField(required=False)
    disk_autoupload = db.BooleanField(required=False, default=False)
    allowed_gifts = db.ListField(db.StringField(choices=ALL_GIFTS), required=False, default=ALL_GIFTS)

    meta = {
        'db_alias': 'advisor',
        'strict': False,
    }

    def __unicode__(self):
        return u'Phone: {}'.format(self.pk)

    def clean(self):
        phone_id = self.calculate_phone_id(imei=self.imei, serial_no=self.serial_number,
                                           wifi_mac=self.wifi_mac, bt_mac=self.bluetooth_mac)
        if phone_id != self.id:
            raise db.ValidationError('Phone_id is not correct')

    @staticmethod
    def calculate_phone_id(imei, serial_no, wifi_mac, bt_mac):
        return sha256(str(imei[:14] + serial_no + wifi_mac + bt_mac).upper()).hexdigest()


# noinspection PyClassHasNoInit
class PlusPromocode(db.Document):
    value = db.StringField(required=True)
    passport_uid = db.LongField(required=False)

    meta = {
        'db_alias': 'advisor',
        'strict': False,
    }

    def __unicode__(self):
        return u'Promocode {} (uid={})'.format(self.value, self.passport_uid)


PLUS_AUTO_SUBSCRIPTION = 'plus_auto_subscription'
PLUS_MANUAL_SUBSCRIPTION = 'plus_manual_subscription'
PLUS_PROMOCODE = 'plus_promocode'
NO_GIFTS = 'no_gifts'

PLUS_ONLY_PREFIX = 'plus_only_'
PLUS_ONLY_PLUS_AUTO_SUBSCRIPTION = PLUS_ONLY_PREFIX + PLUS_AUTO_SUBSCRIPTION
PLUS_ONLY_PLUS_MANUAL_SUBSCRIPTION = PLUS_ONLY_PREFIX + PLUS_MANUAL_SUBSCRIPTION
PLUS_ONLY_PLUS_PROMOCODE = PLUS_ONLY_PREFIX + PLUS_PROMOCODE


# noinspection PyClassHasNoInit
class GreetingMail(db.Document):
    passport_uid = db.LongField(required=True)
    plus_promocode = db.StringField(required=False)
    taxi_promocode = db.StringField(required=False)
    mail_type = db.StringField(
        required=True,
        choices=(
            PLUS_AUTO_SUBSCRIPTION,
            PLUS_MANUAL_SUBSCRIPTION,
            PLUS_PROMOCODE,
            NO_GIFTS,

            # https://st.yandex-team.ru/ADVISOR-2274
            PLUS_ONLY_PLUS_AUTO_SUBSCRIPTION,
            PLUS_ONLY_PLUS_MANUAL_SUBSCRIPTION,
            PLUS_ONLY_PLUS_PROMOCODE,
        )
    )
    phone_id = db.StringField(required=True)

    meta = {
        'db_alias': 'advisor',
        'strict': False,
    }

    def __unicode__(self):
        return u"Mail '{}' for uid={}".format(self.mail_type, self.passport_uid)
