from mongoengine import DENY

from jafar.feed.recommenders import ArrangerRecommender
from jafar.mongo_configs.feed import MongoConfig, db, FeedRecommenderConfig
from jafar.pipelines.predefined import predefined_arranger_pipelines

APP_ITEM_TYPE = u'app'
WIDGET_ITEM_TYPE = u'widget'
PLACEHOLDER_TYPE = u'placeholder'
DEFAULTS_TYPE = u'default_category'


class Position(db.EmbeddedDocument):
    x = db.IntField(required=True)
    y = db.IntField(required=True)


class ScreenItem(db.EmbeddedDocument):
    cell = db.EmbeddedDocumentField(Position, required=True)
    span = db.EmbeddedDocumentField(Position, required=True, default=Position(1, 1))
    item_type = db.StringField(choices=(APP_ITEM_TYPE, WIDGET_ITEM_TYPE, PLACEHOLDER_TYPE, DEFAULTS_TYPE),
                               default=APP_ITEM_TYPE,
                               verbose_name='Type of item',
                               help_text='constant or placeholder for arranger ranking')
    value = db.StringField(required=False, verbose_name='package_name / class_name')
    rearrange = db.BooleanField(default=True, verbose_name='Rearrange', help_text='Rearrange items in this block')


class ScreenTemplate(db.EmbeddedDocument):
    items = db.ListField(db.EmbeddedDocumentField(ScreenItem))


class ArrangerRecommenderConfig(FeedRecommenderConfig):
    recommender = ArrangerRecommender
    pipeline = db.StringField(required=True, choices=predefined_arranger_pipelines.keys())


class PriorityGroup(MongoConfig):
    apps = db.ListField(db.StringField(), verbose_name='List of apps in priority order')
    boost = db.BooleanField(default=False, verbose_name='Pin this group to the top')


class ArrangerExperimentConfig(MongoConfig):
    recommender_config = db.ReferenceField(ArrangerRecommenderConfig, reverse_delete_rule=DENY,
                                           verbose_name='Recommender config')
    dock_template = db.EmbeddedDocumentField(ScreenTemplate, required=True, verbose_name='Dock template')
    screens_template = db.ListField(db.EmbeddedDocumentField(ScreenTemplate, required=False),
                                    verbose_name='Homescreen templates')
    priority_groups = db.ListField(db.ReferenceField(PriorityGroup, reverse_delete_rule=DENY))

    def get_params(self):
        result = super(ArrangerExperimentConfig, self).get_params()
        result['priority_groups'] = [PriorityGroup.objects.get(id=g).get_params() for g in result['priority_groups']]
        result['recommender_config'] = self.recommender_config
        return result
