from collections import defaultdict

import numpy as np
from flask import escape, current_app as app
from mongoengine import PULL

from jafar import db, localization_mongo, fast_cache
from jafar.feed import recommenders, strategy as feed_strategy
from jafar.mongo_configs.base import MongoConfig
from jafar.mongo_configs.card_configs import CardMongoConfig
from jafar.pipelines.predefined import predefined_pipelines, predefined_gifts_pipelines
from jafar.utils import get_all_subclasses

ALL_CATEGORIES_ID = 'ALL_AVAILABLE_CATEGORIES'


class TitlePair(db.EmbeddedDocument):
    title = db.StringField()
    subtitle = db.StringField()


class FeedRecommenderConfig(MongoConfig):
    title_pairs = db.ListField(db.EmbeddedDocumentField(TitlePair), verbose_name='Title pairs')
    default_categories = db.ListField(db.StringField(), default=[ALL_CATEGORIES_ID],
                                      help_text='Use "%s" id for all default categories. '
                                                'Empty list means NOT to generate recommendations '
                                                'if no other mentioned.' % ALL_CATEGORIES_ID)
    top_n = db.IntField(verbose_name='Top N', default=None,
                        help_text='Leave it empty for default settings.')

    meta = {
        'allow_inheritance': True,
        'strict': False
    }

    def validate(self, clean):
        translations = localization_mongo.db[app.config['LAUNCHER_TRANSLATIONS_COLLECTION']]

        if not self.title_pairs:
            raise db.ValidationError('title_pairs must not be empty')
        for title_pair in self.title_pairs:
            for key in (title_pair.title, title_pair.subtitle):
                if translations.count({'_id': key}) == 0:
                    raise db.ValidationError('Key "%s" not found in translations db' % key)
        if len(self.title_set) != len(self.title_pairs):
            raise db.ValidationError("titles must be unique")
        if self.id:  # modifying existing config
            for experiment in FeedExperimentConfig.objects.filter(recommenders__contains=self):
                for other_recommender in experiment.recommenders:
                    if other_recommender == self:
                        continue
                    common_titles = other_recommender.title_set.intersection(self.title_set)
                    if common_titles:
                        raise db.ValidationError(
                            "This change breaks experiment config {}, the following titles are not unique: {}".format(
                                experiment.name, list(common_titles)
                            )
                        )

    @property
    def is_used(self):
        return any(self in experiment.get_recommender_configs()
                   for experiment in FeedExperimentConfig.objects.no_dereference())

    @property
    def title_set(self):
        return set([pair.title for pair in self.title_pairs])

    def get_params(self):
        # feed recommenders need to know their name
        params = super(FeedRecommenderConfig, self).get_params()
        params['experiment_name'] = self.name
        if ALL_CATEGORIES_ID in params['default_categories']:
            params['default_categories'] = None  # pipelines treat None as "use all available categories"
        return params


class SimpleLocalFeedRecommenderConfig(FeedRecommenderConfig):
    recommender = recommenders.SimpleLocalFeedRecommender
    pipeline = db.StringField(
        verbose_name='Pipeline',
        choices=predefined_pipelines.keys(),
        default='local'
    )


class PlaceholdersPromoFeedRecommenderConfig(FeedRecommenderConfig):
    recommender = recommenders.PlaceholdersPromoFeedRecommender

    pipeline = db.StringField(verbose_name='Reserve items pipeline', choices=predefined_pipelines.keys())
    placement_id = db.StringField(verbose_name='External promo provider placement ID')
    external_promo_provider = db.StringField(verbose_name='External promo provider', choices=('direct',))
    external_promo_ttl = db.IntField(verbose_name='Promo TTL')


class SimilarItemsRecommenderConfig(FeedRecommenderConfig):
    recommender = recommenders.SimilarItemsRecommender
    pipeline = db.StringField(
        verbose_name='Pipeline',
        choices=recommenders.SimilarItemsRecommender.pipeline_choices
    )


class TopNRecommenderConfig(FeedRecommenderConfig):
    recommender = recommenders.TopNRecommender
    pipeline = db.StringField(
        verbose_name='Pipeline',
        choices=predefined_pipelines.keys()
    )


class GiftsRecommenderConfig(FeedRecommenderConfig):
    recommender = recommenders.GiftsRecommender
    max_view_count = db.IntField(verbose_name='Max view count', default=None,
                                 help_text='Limit of one gift show number (in 2 weeks).')
    disliked_items_ttl = db.IntField(verbose_name='Dislike TTL', default=None, required=True, min_value=0,
                                     help_text='In seconds')
    pipeline = db.StringField(
        verbose_name='Pipeline',
        choices=predefined_gifts_pipelines.keys()
    )


class TopNCategorizedRecommenderConfig(TopNRecommenderConfig):
    recommender = recommenders.TopNCategorizedRecommender


class CardPreference(db.EmbeddedDocument):
    card_config = db.ReferenceField(CardMongoConfig, required=True)
    weight = db.FloatField(default=0.0)


class StrategyConfig(MongoConfig):
    CONSTRAINT_CHOICES = [
        (constraint.__name__, constraint.hint)
        for constraint in get_all_subclasses(feed_strategy.CustomConstraint)
    ]

    constraints = db.ListField(
        db.StringField(choices=CONSTRAINT_CHOICES, verbose_name='Constraints'),
        help_text=('Note: OneHotPositionConstraint and RecommendationGroupCapacityConstraint are '
                   'mandatory and will be applied in any case')
    )
    randomization = db.StringField(
        choices=('0.0', '1.0'), default='1.0', verbose_name='Randomization',
        help_text=escape('Controls the degree of randomization for objective fuction, '
                         'adding <this> * rand(-1, 1) to each objective weight. '
                         'Setting this to 0 will disable randomization (not recommended)')
    )
    card_configs = db.ListField(
        db.EmbeddedDocumentField(CardPreference), verbose_name='Card configs',
        help_text=('Cards allowed in this strategy, weighed by preference. '
                   'Weight will be used as an objective function coefficient '
                   'applied to corresponding card variables.')
    )

    meta = {
        'strict': False
    }

    def get_params(self):
        params = super(StrategyConfig, self).get_params()
        constraints = []
        for constraint_class in self.constraints:
            try:
                constraints.append(getattr(feed_strategy, constraint_class)())
            except AttributeError:
                pass
        card_configs = []
        card_weights = []
        for item in self.card_configs:
            card_configs.append((item.card_config.card_config_class, item.card_config.get_params()))
            card_weights.append(item.weight)

        params['strategy'] = feed_strategy.Strategy(
            constraints=constraints,
            card_configs=card_configs,
            card_weights=np.array(card_weights),
            randomization=float(params['randomization'])
        )
        return params


class WeightedRecommenderConfig(db.EmbeddedDocument):
    recommender_config = db.ReferenceField(FeedRecommenderConfig, verbose_name='Recommender config')
    weight = db.FloatField(
        default=0.0, verbose_name='Weight',
        help_text=(
            'NOTE: negative weights are treated as penalties: BalancedRecommenderConstraint '
            'if not applied to these recommenders.'
        )
    )


@fast_cache.memoize(timeout=5 * 60)
def get_active_experiments():
    experiments = localization_mongo.db[app.config['LAUNCHER_EXPERIMENTS_COLLECTION']]
    return frozenset(
        item['value']
        for doc in experiments.find({'_id': {'$regex': '.*_experiment$'}})
        for item in doc['values']
    )


class FeedExperimentConfig(MongoConfig):
    strategy = db.ReferenceField(StrategyConfig, verbose_name='Strategy', required=True)
    block_count = db.IntField(min_value=1, verbose_name='Block count', required=True)
    lifetime_seconds = db.IntField(min_value=0, required=True, verbose_name='Feed lifetime in sec')
    page_limit = db.IntField(min_value=0, verbose_name='Page limit (0 corresponds to unlimited)', default=0)
    item_filter_size = db.IntField(
        min_value=0, required=True, default=2,
        verbose_name='Item filter size (in pages)'
    )
    explanation_filter_size = db.IntField(
        min_value=0, required=True, default=5,
        verbose_name='Explanation filter size (in pages)')
    recommenders = db.ListField(
        db.ReferenceField(FeedRecommenderConfig, reverse_delete_rule=PULL),
        verbose_name='Feed recommender configs'
    )
    weighted_recommenders = db.ListField(
        db.EmbeddedDocumentField(WeightedRecommenderConfig),
        verbose_name='Feed recommender configs'
    )

    def validate(self, clean):
        title_dict = defaultdict(list)
        for recommender_config in self.recommenders:
            for title in recommender_config.title_set:
                title_dict[title].append(recommender_config.name)

        if not all((len(value) == 1 for value in title_dict.values())):
            raise db.ValidationError(
                "titles must be unique across recommenders: {}".format(
                    ', '.join([
                        "{} is present in {}".format(title, recommenders)
                        for title, recommenders in title_dict.iteritems()
                        if len(recommenders) > 1
                    ])
                )
            )

    def get_recommender_configs(self):
        if self.weighted_recommenders:
            return [item.recommender_config for item in self.weighted_recommenders]
        else:
            return self.recommenders

    def get_recommender_params(self):
        # NOTE: combatibility logic to allow both `recommenders`
        # and `weighted_recommenders` to work
        use_weighted = bool(self.weighted_recommenders)
        result = []
        for item in (self.weighted_recommenders or self.recommenders):
            if use_weighted:
                config = item.recommender_config
                params = {'weight': item.weight}
            else:
                config = item
                params = {}
            params.update(config.get_params())
            result.append((config.recommender, params))
        return result

    def get_params(self):
        params = super(FeedExperimentConfig, self).get_params()
        params['strategy'] = self.strategy.get_params()['strategy']
        params['recommenders'] = self.get_recommender_params()
        del params['weighted_recommenders']
        # feed experiments also need to know their name
        params['experiment_name'] = self.name
        params['page_limit'] = self.page_limit
        return params

    @property
    def is_used(self):
        return self.name in get_active_experiments()

    meta = {
        'indexes': [
            'name'
        ],
        'index_background': True,
        'strict': False
    }
