from jafar.storages import make_key


class BaseBlock(object):

    def __init__(self, input_data=None, output_data=None, destroyed_data=None, key_prefix=None, key_country_override=None):
        """
        :param input_data: frame keys required by this block to function
        :param output_data: new frame keys this block will add to context
        :param destroyed_data: frame keys this block will delete from context,
                               because they are no longer required.
        :param key_prefix: prefix for storage key
        :param key_country_override: key country (overrides context country if not None)

        These values are used to verify pipeline consistency at initialization.
        Some blocks which can accept different kinds of input frames (for example,
        target blocks may require different kinds of user-item install data from
        different datasets) will also use `input_data` parameter to determine which
        frame they will use as a data source.
        """
        self.input_data = input_data or []
        self.output_data = output_data or []
        self.destroyed_data = destroyed_data or []
        self.key_prefix = key_prefix
        self.key_country_override = key_country_override

    def key_for(self, context, name=None):
        country = context.country if self.key_country_override is None else self.key_country_override
        key_args = [context.pipeline.name, country]
        if self.key_prefix:
            key_args.append(self.key_prefix)
        if name:
            key_args.append(name)
        return make_key(*key_args)

    def set_params(self, params):
        for key, value in params.iteritems():
            setattr(self, key, value)


class SingleContextBlock(BaseBlock):
    """
    Type of blocks that can handle pipeline contexts separately one by one
    """

    def apply(self, context, train):
        raise NotImplementedError
