import logging

from jafar.storages import make_key
from jafar.storages.memory import MemoryStorage

logger = logging.getLogger(__name__)


class PipelineContext(object):
    """
    Context:
    - Provides pipeline's blocks with data specific for countries/categories
    - It is a way to pass data from one block to another
    - It is also a mean to query a pipeline: it can contain specific ("target")
      frames which we're asking to evaluate, or specific categories we want to
      get recommendations from.
    """

    def __init__(self, pipeline, country, requested_categories=None, default_categories=None, data=None):
        """
        :param pipeline: a Pipeline object
        :param country: is used to load country-specific datasets and models
                        (trained on those datasets)
        :param requested_categories: if present, specifically tells the pipeline to generate
                           recommendations from the following categories only.
                           Otherwise, use default_categories.
        :param default_categories: fallback categories. Empty list means nothing to recommend.
                            None means all categories.
        :param data: dictionary containing dataframes and other objects passing
                     through the pipeline.
        """
        self.pipeline = pipeline
        self.country = country
        self.requested_categories = requested_categories
        self.default_categories = default_categories
        self.data = data or {}

    @property
    def storage(self):
        return self.pipeline.storage

    @property
    def n_users(self):
        return self.storage.get_proxy(make_key(self.pipeline.name, self.country, 'n_users'))[0]

    @property
    def n_items(self):
        return self.storage.get_proxy(make_key(self.pipeline.name, self.country, 'n_items'))[0]

    def __hash__(self):
        return hash((
            (self.country,) + tuple(self.requested_categories or ())
        ))

    def __eq__(self, other):
        return (
            self.country == other.country and
            self.requested_categories == other.requested_categories
        )

    def __str__(self):
        return "<PipelineContext: country={} requested categories={}>".format(
            self.country, self.requested_categories
        )

    def copy(self, with_data=True):
        return PipelineContext(
            self.pipeline,
            self.country,
            self.requested_categories,
            data={key: value.copy() for key, value in self.data.iteritems()} if with_data else {}
        )

    def free(self):
        logger.debug('Deleting data from %s-%s context', self.country, self.requested_categories)
        self.data = {}


class TemporaryPipelineContext(PipelineContext):
    def __init__(self, *args, **kwargs):
        super(TemporaryPipelineContext, self).__init__(*args, **kwargs)
        self.temporary_storage = MemoryStorage()

    @property
    def storage(self):
        return self.temporary_storage

    def copy(self):
        raise NotImplementedError


class EmptyPipelineContext(object):
    pass
