from jafar.pipelines import Pipeline, ids
from jafar.pipelines.blocks import data, selection, discount, filtering, features


class BaseGiftsPipelineFactory(object):
    name = NotImplemented
    selection_block_cls = NotImplemented

    def __call__(self, pipeline_config, storage, top_n, promo_provider=None, **kwargs):
        self.top_n = top_n
        blocks = list(self.get_blocks())
        return Pipeline(blocks, name=self.name, storage=storage)

    def get_blocks(self):
        yield 'read_user_features', data.OnlineReadDataBlock()
        yield 'read_usage_stats', data.OnlineUsageStatsData(counters=['rec_view_count'])
        yield 'select_gifts', self.selection_block_cls()
        yield 'constant_feature', features.ConstantFeatureBlock(value=1., input_frame=ids.FRAME_KEY_PREDICTIONS,
                                                                feature_name='value')
        yield 'impression_discount', discount.ImpressionDiscountBlock()
        yield 'filter_banned', filtering.BannedItemsFilteringBlock(blacklist_component='recommendations')
        yield 'filter_basket', filtering.InstallsFilteringBlock()
        yield 'leave_top', filtering.LeaveTopBlock(top_n=self.top_n, input_frame=ids.FRAME_KEY_PREDICTIONS)


class SantaPipelineFactory(BaseGiftsPipelineFactory):
    name = 'santa'
    selection_block_cls = selection.AvailableGiftsSelectionBlock


class RudolphPipelineFactory(BaseGiftsPipelineFactory):
    name = 'rudolph'
    selection_block_cls = selection.YandexDistributedAppSelectionBlock


create_santa_pipeline = SantaPipelineFactory()
create_rudolph_pipeline = RudolphPipelineFactory()
