import os
import unittest
import mongomock
import mongoengine
import numpy as np
import random

from flask import current_app as app

from jafar import create_app
from jafar import jafar_mongo
from jafar import advisor_mongo
from jafar import advisor_replica_mongo
from jafar.utils.cmph_wrapper import set_seed


def reset_random_seed():
    np.random.seed(1234)
    random.seed(1234)
    set_seed(1234)


class JafarTestCase(unittest.TestCase):

    @classmethod
    def create_app(cls, *args, **kwargs):
        config_overrides = {
            'TESTING': True,
            'MEMMAP_STORAGE_DIR': '/tmp/jafar-temp-memmap-storage',
            'MEMMAP_STORAGE_MODE': 'lazy'
        }
        app = create_app(**config_overrides)

        from jafar.storages.memory import MemoryStorage
        from jafar.storages.memmap import MemmapStorage

        with app.app_context():
            cls.check_mongomock()
            cls.memory_storage = MemoryStorage()
            cls.memmap_storage = MemmapStorage()

        return app

    @classmethod
    def check_mongomock(cls):
        # check pymongo connections
        for connection in (
                jafar_mongo,
                advisor_mongo,
                advisor_replica_mongo
        ):
            assert isinstance(connection.cx, mongomock.MongoClient), \
                "Connection to {} is not using mongomock!".format(connection.db.name)

        # check mongoengine default connection
        mongoengine_connection = mongoengine.connection.get_connection()
        assert isinstance(mongoengine_connection, mongomock.MongoClient), \
            "Mongoengine default connection is not using mongomock!"

        mongoengine_advisor_connection = mongoengine.connection.get_connection('advisor')
        assert isinstance(mongoengine_advisor_connection, mongomock.MongoClient), \
            "Mongoengine 'advisor' connection is not using mongomock!"

        # check that mongoengine's connection is the same as jafar_mongo
        assert jafar_mongo.cx is mongoengine_connection, \
            "jafar_mongo and mongoengine default connection are not the same!"
        assert advisor_mongo.cx is mongoengine_advisor_connection, \
            "advisor_mongo and mongoengine 'advisor' connection are not the same!"

    @classmethod
    def clear_memmap_storage(cls):
        memmap_dir = app.config['MEMMAP_STORAGE_DIR']
        assert memmap_dir == '/tmp/jafar-temp-memmap-storage'
        for filename in os.listdir(memmap_dir):
            os.remove(os.path.join(memmap_dir, filename))

    def setUp(self):
        self.app = self.create_app()
        self.client = self.app.test_client()
        self._ctx = self.app.test_request_context()
        self._ctx.push()
        reset_random_seed()

    def tearDown(self):
        advisor_mongo.cx.drop_database(advisor_mongo.db)
        self._ctx.pop()
        del self._ctx
        del self.client
        del self.app
