from jafar.admin.models.wallpapers import WallpaperCategories

from jafar.tests import JafarTestCase
from jafar.utils import to_ascii

import json
import mock

TEST_CATEGORY_NAME = 'Test category name'
COVER_TITLE_KEY = 'backend_categories_auto_and_vehicles'
COVER_TITLE_COLOR = '#ff00c5'
COVER_SUBTITLE_KEY = 'backend_categories_books_and_reference'
COVER_SUBTITLE_COLOR = '#ff4900'
COVER_BACKGROUND_IMAGE_KEY = 'wallpapers/cover_backgrounds/SSTTK_09443ac079f49ee07bb504150d826c6d.jpg'
COVER_BACKGROUND_IMAGE_WIDTH = 1920
COVER_BACKGROUND_IMAGE_HEIGHT = 758
COVER_BACKGROUND_COLOR = '#00ff1a'
COVER_BACKGROUND_HAS_GRADIENT = True
COVER_BACKGROUND_FULL_SIZE = False


class IntegrationTestCase(JafarTestCase):

    def _build_category(self, composite_cover_background_image_is_none=False):
        data = {
            'name': TEST_CATEGORY_NAME,
            'composite_cover': {
                'title': {
                    'text': COVER_TITLE_KEY,
                    'color': COVER_TITLE_COLOR
                },
                'subtitle': {
                    'text': COVER_SUBTITLE_KEY,
                    'color': COVER_SUBTITLE_COLOR
                },
                'background': {
                    'color': COVER_BACKGROUND_COLOR,
                    'has_gradient': COVER_BACKGROUND_HAS_GRADIENT
                },
                'full_size': COVER_BACKGROUND_FULL_SIZE
            },
            'images': [],
            'use_in_autochange': False,
            'use_in_feed': False,
            'updated_at': {
                '$date': 1573064562392
            }
        }

        if not composite_cover_background_image_is_none:
            data.update({'composite_cover': {
                'background': {
                    'image': {
                        'key': COVER_BACKGROUND_IMAGE_KEY,
                        'width': COVER_BACKGROUND_IMAGE_WIDTH,
                        'height': COVER_BACKGROUND_IMAGE_HEIGHT,
                        'hash': '09443ac079f49ee07bb504150d826c6d'
                    }}
            }})

        return json.dumps(data)

    @mock.patch('jafar.admin.ajax.TranslationsAjaxLoader.get_translation_list',
                side_effect=lambda key: {'en': '1'} if key == COVER_TITLE_KEY else {'en': '2'})
    @mock.patch('jafar.admin.models.wallpapers.LANGUAGE_CHOICES', ('en',))
    def test_generate_preview_preview_is_1_image(self, ajax_loader):
        category = WallpaperCategories.from_json(self._build_category())

        assert len(category._generate_previews_for_composite_cover()) == 1

    @mock.patch('jafar.admin.ajax.TranslationsAjaxLoader.get_translation_list',
                side_effect=lambda key: {'en': '1'} if key == COVER_TITLE_KEY else {'en': '2'})
    @mock.patch('jafar.admin.models.wallpapers.LANGUAGE_CHOICES', ('en', 'ru', 'be', 'uk', 'kz'))
    def test_generate_preview_for_languages_less_than_required_1_image_generated(self, ajax_loader):
        """
        jafar.admin.models.wallpapers.LANGUAGE_CHOICES - required languages, has 5 elements,
        But ajax loader returns only 1

        Only 1 image will be generated
        """
        category = WallpaperCategories.from_json(self._build_category())

        assert len(category._generate_previews_for_composite_cover()) == 1

    @mock.patch('jafar.admin.ajax.TranslationsAjaxLoader.get_translation_list',
                side_effect=lambda key: {'en': '1', 'ru': '1_ru'} if key == COVER_TITLE_KEY else {
                    'en': '2', 'ru': '2_ru'})
    @mock.patch('jafar.admin.models.wallpapers.LANGUAGE_CHOICES', ('en', 'ru', 'be', 'uk', 'kz'))
    def test_generate_preview_for_languages_less_than_required_2_image_generated(self, ajax_loader):
        """
        jafar.admin.models.wallpapers.LANGUAGE_CHOICES - required languages, has 5 elements,
        But ajax loader returns 2

        Only 2 images will be generated
        """
        category = WallpaperCategories.from_json(self._build_category())

        assert len(category._generate_previews_for_composite_cover()) == 2

    @mock.patch('jafar.admin.ajax.TranslationsAjaxLoader.get_translation_list',
                side_effect=lambda key: {'en': '1', 'ru': '1_ru'} if key == COVER_TITLE_KEY else {
                    'en': '2', 'ru': '2_ru'})
    @mock.patch('jafar.admin.models.wallpapers.LANGUAGE_CHOICES', ('en', 'ru', 'be', 'uk', 'kz'))
    def test_generate_preview_for_languages_equal_to_required_5_image_generated(self, ajax_loader):
        """
        jafar.admin.models.wallpapers.LANGUAGE_CHOICES - required languages, has 5 elements,
        ajax loader returns 5

        5 images will be generated
        """
        category = WallpaperCategories.from_json(self._build_category())

        assert len(category._generate_previews_for_composite_cover()) == 2

    @mock.patch('jafar.admin.ajax.TranslationsAjaxLoader.get_translation_list',
                side_effect=lambda key: {'uk': '1', 'ru': '1_ru'} if key == COVER_TITLE_KEY else {
                    'uk': '2', 'ru': '2_ru'})
    @mock.patch('jafar.admin.models.wallpapers.LANGUAGE_CHOICES', ('en', 'ru'))
    def test_generate_preview_required_languages_has_different_items_1_image_generated(self, ajax_loader):
        """
        jafar.admin.models.wallpapers.LANGUAGE_CHOICES - has 2 lements: 'ru', 'en'
        ajax loader returns 2 elements: 'ru', 'uk'

        only image for 'ru' will be generated
        """
        category = WallpaperCategories.from_json(self._build_category())

        assert len(category._generate_previews_for_composite_cover()) == 1

    @mock.patch('jafar.admin.ajax.TranslationsAjaxLoader.get_translation_list', side_effect=lambda key: None)
    @mock.patch('jafar.admin.models.wallpapers.LANGUAGE_CHOICES', ('en', 'ru'))
    def test_generate_preview_no_text_obtained_2_images_generated(self, ajax_loader):
        """
        jafar.admin.models.wallpapers.LANGUAGE_CHOICES - has 2 lements: 'ru', 'en'
        ajax loader returns 0 elements

        generated 2 images for both locales
        """
        category = WallpaperCategories.from_json(self._build_category())

        assert len(category._generate_previews_for_composite_cover()) == 1

    @mock.patch('jafar.admin.ajax.TranslationsAjaxLoader.get_translation_list',
                side_effect=lambda key: {'ru': '1_ru'} if key == COVER_TITLE_KEY else {'ru': '2_ru'})
    @mock.patch('jafar.admin.models.wallpapers.LANGUAGE_CHOICES', ('ru',))
    def test_generate_preview_background_image_not_specified_0_images_generated(self, ajax_loader):
        """
        composite_cover.background.image - is required parameter, don't generate images without it
        """
        category = WallpaperCategories.from_json(
            self._build_category(composite_cover_background_image_is_none=True))

        assert len(category._generate_previews_for_composite_cover()) == 0

    @mock.patch('jafar.admin.ajax.TranslationsAjaxLoader.get_translation_list',
                side_effect=lambda key: {'ru': '1_ru'} if key == COVER_TITLE_KEY else {'ru': '2_ru'})
    @mock.patch('jafar.admin.models.wallpapers.LANGUAGE_CHOICES', ('ru',))
    def test_generate_preview_background_image_not_specified_0_images_generated(self, ajax_loader):
        """
        composite_cover.background.image - is required parameter, don't generate images without it
        """
        category = WallpaperCategories.from_json(
            self._build_category(composite_cover_background_image_is_none=True))

        assert len(category._generate_previews_for_composite_cover()) == 0

    @mock.patch('jafar.admin.models.wallpapers.LANGUAGE_CHOICES', ('ru',))
    @mock.patch('PIL.Image')
    @mock.patch('jafar.s3_client')
    def test_save_generated_image_to_s3_added_1_image_to_generated_covers(self, mock_image, s3_client):
        generated_images = {'ru': mock_image}
        category = WallpaperCategories.from_json(
            self._build_category(composite_cover_background_image_is_none=True))

        category._save_generated_covers_to_s3(generated_images)

        assert len(category.composite_cover.generated_covers) == 1

    @mock.patch('jafar.admin.models.wallpapers.LANGUAGE_CHOICES', ('ru',))
    @mock.patch('PIL.Image')
    @mock.patch('jafar.s3_client')
    def test_save_generated_image_to_s3_language_is_ru(self, mock_image, s3_client):
        generated_images = {'ru': mock_image}
        category = WallpaperCategories.from_json(
            self._build_category(composite_cover_background_image_is_none=True))

        category._save_generated_covers_to_s3(generated_images)

        assert category.composite_cover.generated_covers[0].language == 'ru'

    @mock.patch('jafar.admin.models.wallpapers.LANGUAGE_CHOICES', ('ru',))
    @mock.patch('PIL.Image')
    @mock.patch('jafar.s3_client')
    def test_save_generated_image_to_s3_url_is_correct(self, mock_image, s3_client):
        generated_images = {'ru': mock_image}
        category = WallpaperCategories.from_json(
            self._build_category(composite_cover_background_image_is_none=True))

        category._save_generated_covers_to_s3(generated_images)

        expected_url = 'https://lnchr-files.s3.mdst.yandex.net/cover_Test category name_ru.jpg'
        assert category.composite_cover.generated_covers[0].image_full.url == expected_url


