import cPickle as pickle
import json
import os
import unittest
from collections import defaultdict
from datetime import datetime
from urllib import urlencode
from uuid import UUID

from flask import current_app as app
from mock import patch

from jafar import advisor_mongo, storage_wrapper
from jafar.commands.train.train_recommenders import TrainRecommenders
from jafar.data_providers.launcher.blacklist import blacklist_caches
from jafar.feed.schema import DATETIME_FORMAT
from jafar.mongo_configs.blacklist import Blacklist, Distributor
from jafar.tests import JafarTestCase, reset_random_seed
from jafar.tests.fixtures import mongo_configs as config_fixtures, vanga as vanga_fixtures
from jafar.tests.fixtures.profile import (fake_user_profile, user_id, update_user_profile, passport_uid, apps,
                                          bad_user_id)
from jafar.tests.fixtures.setup_wizard import fake_gifts, drop_gifts
from jafar.tests.mocks.datasets import mock_get_dataset_processor, dummy_memoize
from jafar.vanga.stats_getter import StatsContext


@patch('jafar.clickhouse._execute', lambda *args: [])
class IntegrationTestCase(JafarTestCase):
    default_request_params = {
        "view_type": "feed",
        "user_info": {
            "card_configs": [{"card_type": "Scrollable_Expandable", "count": 4},
                             {"card_type": "Bonus_universal_card", "count": 2}],
            "device_id": user_id,
            "passport_uid": passport_uid,
            "clids": {
                "clid1": "2246892",
                "clid1006": "2246894",
            }
        },
        "place": "feed", "page": 1,
        "google_categories": ["SOCIAL"]
    }

    @classmethod
    @patch('jafar.pipelines.blocks.data.get_dataset_processor', mock_get_dataset_processor)
    def setUpClass(cls):
        # train recommenders once per testcase
        with cls.create_app().app_context():
            pipelines = ['kano', 'sonya']

            for pipeline in pipelines:
                config_fixtures.fake_pipeline_recommender(pipeline, pipeline)
                reset_random_seed()
                TrainRecommenders().run(pipeline=pipeline, country='RU', recommendation_mode='generate')

            config_fixtures.fake_gift_recommender('santa', 'santa')

    @classmethod
    def tearDownClass(cls):
        # clean up storage
        with cls.create_app().app_context():
            cls.clear_memmap_storage()

    def setUp(self):
        super(IntegrationTestCase, self).setUp()
        storage_wrapper.storage.load_meta()
        fake_user_profile()
        fake_gifts()

        # We do not train US recommender in this test, so set default country as RU
        app.config['DEFAULT_COUNTRY'] = 'RU'

    def tearDown(self):
        advisor_mongo.db.profile.drop()
        drop_gifts()
        super(IntegrationTestCase, self).tearDown()

    def get_estimator_param_fixture(self, name):
        path = os.path.join(app.config['BASE_DIR'], 'fixtures', 'estimator_params', name)
        with open(path) as f:
            return pickle.load(f)

    def make_request(self, url, params=None, response_code=200):
        params = params or self.default_request_params
        response = self.client.post(url, data=json.dumps(params))
        self.assertEqual(response.status_code, response_code, response.data)
        response = json.loads(response.data)
        return response

    def compare_response(self, response, expected, field_name='package_name'):
        self.assertIn('blocks', response)
        self.assertIsInstance(response['blocks'], list)
        expected_dict = {}
        for block in expected:
            for item in block:
                expected_dict[item[field_name]] = item
        response_dict = {}
        for block in response['blocks']:
            for item in block['items']:
                response_dict[item[field_name]] = item
        self.assertEquals(sorted(expected_dict.keys()), sorted(response_dict.keys()))
        for package_name, expected_item in expected_dict.iteritems():
            actual_item = response_dict[package_name]
            for k, v in actual_item.iteritems():
                if k in expected_item:
                    if isinstance(v, float):
                        self.assertAlmostEqual(v, expected_item[k], 5)
                    else:
                        self.assertEqual(v, expected_item[k])

    def check_promo_count(self, response, asserted_count):
        result_promo_count = 0
        for item in response['blocks'][0]['items']:
            if 'offer_id' in item:
                result_promo_count += 1
        self.assertEqual(result_promo_count, asserted_count)

    """
    Tests start here.
    NOTE: this is an integration test, therefore test order matters
    (mostly because recommendations are shuffled randomly and np.random.seed
    only helps if test order is fixed). Order is enforced by 'test_%number%'
    prefix; you should follow this notation when adding new tests.
    """

    def test_01_impression_discount_block_usage(self):
        with patch('jafar.pipelines.blocks.discount.ImpressionDiscountBlock.apply',
                   side_effect=lambda context, train: context) as mocked:
            self.make_request('/feed/sonya')
            mocked.assert_called()

    def test_02_impression_discount_block_usage(self):
        with patch('jafar.pipelines.blocks.discount.ImpressionDiscountBlock.apply',
                   side_effect=lambda context, train: context) as mocked:
            self.make_request('/feed/kano')
            mocked.assert_called()

    def test_03_recommendation_requests(self):
        response = self.make_request('/feed/sonya')
        self.compare_response(response, [[
            {'package_name': 'tv.shou.android'},
            {'package_name': 'jp.naver.lineplay.android'},
            {'package_name': 'com.askfm'},
            {'package_name': 'com.redcactus.repost'},
        ]])

    def test_04_recommendation_requests(self):
        response = self.make_request('/feed/kano')
        self.compare_response(response, [[
            {'package_name': 'com.appsgo.loveis'},
            {'package_name': 'dstudio.tool.instasave'},
            {'package_name': 'ru.mail.love'},
            {'package_name': 'ru.rutube.app'},
        ]])

    def test_05_blacklist(self):
        def get_apps():
            blacklist_caches['recommendations'].force_reload()
            response = self.make_request('/feed/kano')
            return [item['package_name'] for block in response['blocks'] for item in block['items']]

        app = get_apps()[0]

        blacklist = Blacklist(
            component='recommendations',
            name='some_name',
            package_names=[app],
            targeting={},
        )
        blacklist.save()
        self.assertNotIn(app, get_apps())

        distributor = Distributor(name='Ya', clid1006='some_other_clid')
        distributor.save()
        blacklist.targeting.distributors = [distributor]
        blacklist.save()

        self.assertIn(app, get_apps())

        distributor.clid1006 = self.default_request_params['user_info']['clids']['clid1006']
        distributor.save()

        self.assertNotIn(app, get_apps())

        Blacklist.objects.delete()
        blacklist_caches['recommendations'].force_reload()

    unknown_user_request_params = request_params = {
        "view_type": "feed",
        "user_info": {
            "card_configs": [{"card_type": "Scrollable_Expandable", "count": 4}],
            "device_id": "0" * 32,
            "clids": {
                "clid1": "2246892",
                "clid1006": "2246894",
            }
        },
        "place": "feed", "page": 1,
        "google_categories": ["SOCIAL"]
    }

    def test_11_unknown_user(self):
        # some recommenders require user info, some don't
        # `kano` includes popular baseline recommender and should return the
        # requested amount of items even for unknown users

        response = self.make_request('/feed/kano', self.unknown_user_request_params)
        self.assertEqual(len(response['blocks']), 1)

    def test_12_unknown_user(self):
        # `sonya`, on the other hand, is pure-item-item recommender and requires
        # user info. without it, the response would contain no apps
        response = self.make_request('/feed/sonya', self.unknown_user_request_params, 404)
        self.assertEqual(response, "Empty recommendations")

    @patch('jafar.pipelines.blocks.data.request_cache.memoize', dummy_memoize)
    def test_13_preinstall_filter(self):
        response = self.make_request('/feed/kano')
        items = [item['package_name'] for item in response['blocks'][0]['items']]
        update_user_profile(user=user_id, add=items, is_system=True)

        response = self.make_request('/feed/kano')
        for item in response['blocks'][0]['items']:
            self.assertNotIn(item['package_name'], items)

    def test_14_expire_at_presence(self):
        response = self.make_request('/feed/kano')
        self.assertIn('expire_at', response)
        datetime.strptime(response['expire_at'], DATETIME_FORMAT)

    @patch('jafar.pipelines.blocks.selection.is_yandex_phone', lambda _: True)
    def test_15_gifts_ok(self):
        response = self.make_request('/feed/santa')
        self.compare_response(response, [[
            {'key': 'plus'},
            {'key': 'taxi', 'code': 'taxi_promocode'}
        ]], field_name='key')


@patch('jafar.clickhouse._execute', lambda *args: [])
class VangaIntegrationTestCase(JafarTestCase):
    maxDiff = None

    @classmethod
    def setUpClass(cls):
        with cls.create_app().app_context():
            config_fixtures.fake_vanga_config('nostradamus')

    def setUp(self):
        super(VangaIntegrationTestCase, self).setUp()
        fake_user_profile()
        patcher_get_any_launch = patch('jafar.utils.vanga_helpers.get_any_launch')
        patcher_get_personal_stats = patch('jafar.vanga.stats_getter.PersonalNoFallbackGroup.get_personal_stats')
        patcher_get_general_stats = patch('jafar.vanga_general_stats_loader.load')
        self.addCleanup(patcher_get_any_launch.stop)
        self.addCleanup(patcher_get_personal_stats.stop)
        self.addCleanup(patcher_get_general_stats.stop)
        self.mock_get_any_launch = patcher_get_any_launch.start()
        self.mock_get_personal_stats = patcher_get_personal_stats.start()
        self.mock_get_general_stats = patcher_get_general_stats.start()
        self.mock_get_any_launch.return_value = []
        self.mock_get_personal_stats.return_value = ([], [])
        self.mock_get_general_stats.return_value = {}

    def tearDown(self):
        advisor_mongo.db.profile.drop()
        super(VangaIntegrationTestCase, self).tearDown()

    def make_request(self, url, params=None, response_code=200, request_method='GET'):
        params = params or {}
        if request_method == 'GET':
            response = self.client.get(url, query_string=urlencode(params))
        elif request_method == 'POST':
            response = self.client.post(url, data=json.dumps(params))
        else:
            raise ValueError('Unknown request methods')
        self.assertEqual(response.status_code, response_code, response.data)
        response = json.loads(response.data)
        return response

    def test_response_200(self):
        self.make_request('/vanga/nostradamus', params={'device_id': user_id})
        good_params = [
            ({'device_id': user_id}, (UUID(user_id), [], [], defaultdict(set), False)),
            ({'device_id': user_id, 'places_blacklist': ['homescreens']},
             (UUID(user_id), [['homescreens']], [], defaultdict(set), False)),
            ({'device_id': user_id, 'places_blacklist': ['homescreens', 'all_apps.list']},
             (UUID(user_id), [['homescreens'], ['all_apps', 'list']], [], defaultdict(set), False)),
            ({'device_id': user_id, 'packages': ['abc', 'com.joom']},
             (UUID(user_id), [], ['abc', 'com.joom'], defaultdict(set), False)),
            ({'device_id': user_id, 'version': 2},
             (UUID(user_id), [], [], defaultdict(set), True))
        ]
        for param_set, expected in good_params:
            self.make_request('/vanga/nostradamus', params=param_set, request_method='POST')
            with patch('jafar.vanga.stats_getter.StatsContext', wraps=StatsContext) as patched:
                self.make_request('/vanga/nostradamus', params=param_set, request_method='POST')
                patched.assert_called_with(*expected)

    def test_response_400(self):
        bad_params = [
            {'device_id': bad_user_id},
            {'device_id': user_id, 'places_blacklist': ['a.b.c']},
            {'places_blacklist': ['a.b.c']},
            {'device_id': user_id, 'places_blacklist': ['b.c']},
            {'device_id': user_id, 'places_blacklist': ['homescreens', 'b.c']},
            {'packages': ['abc']},
            {'version': 1}
        ]
        for param_set in bad_params:
            self.make_request('/vanga/nostradamus', params=param_set, response_code=400)

    def test_usage_stats_ch_personal(self):
        self.mock_get_personal_stats.return_value = vanga_fixtures.personal_stats['value']
        self.mock_get_any_launch.return_value = ['fake_value']

        response = self.make_request('/vanga/nostradamus', params={'device_id': user_id})

        self.mock_get_personal_stats.assert_called()
        self.mock_get_general_stats.assert_not_called()
        self.assertEquals(response, vanga_fixtures.personal_stats['response'])

    def test_usage_stats_ch_general(self):
        self.mock_get_general_stats.return_value = vanga_fixtures.general_stats['value']
        update_user_profile(user=user_id, set_={'created_at': datetime.utcnow()})

        response = self.make_request('/vanga/nostradamus', params={'device_id': user_id})

        self.mock_get_personal_stats.assert_not_called()
        self.mock_get_general_stats.assert_called_with(list(apps))
        self.assertEquals(response, vanga_fixtures.general_stats['response'])

    def test_usage_stats_ch_personal_v2(self):
        self.mock_get_personal_stats.return_value = vanga_fixtures.personal_stats_v2['value']
        self.mock_get_any_launch.return_value = ['fake_value']

        response = self.make_request('/vanga/nostradamus', params={'device_id': user_id, 'version': 2})

        self.mock_get_personal_stats.assert_called()
        self.mock_get_general_stats.assert_not_called()
        self.assertEquals(response, vanga_fixtures.personal_stats_v2['response'])

    def test_usage_stats_ch_general_old(self):
        # old user without launches
        response = self.make_request('/vanga/nostradamus', params={'device_id': user_id})

        self.mock_get_personal_stats.assert_not_called()
        self.mock_get_general_stats.assert_not_called()
        self.assertEquals(response, vanga_fixtures.personal_stats_empty['response'])

    def test_usage_stats_ch_no_profile(self):
        # user without profile
        response = self.make_request('/vanga/nostradamus', params={'device_id': UUID(int=0)})

        self.mock_get_personal_stats.assert_not_called()
        self.mock_get_general_stats.assert_not_called()
        self.assertEquals(response, vanga_fixtures.personal_stats_empty['response'])

    def test_usage_stats_ch_no_apps(self):
        # new user without apps
        update_user_profile(user=user_id, set_={'created_at': datetime.utcnow(),
                                                'installed_apps_info': []})

        response = self.make_request('/vanga/nostradamus', params={'device_id': user_id})

        self.mock_get_personal_stats.assert_not_called()
        self.mock_get_general_stats.assert_called_with([])
        self.assertEquals(response, vanga_fixtures.personal_stats_empty['response'])
