import json
import logging
import os

import numpy as np
import pandas as pd
from flask import current_app as app

from jafar.utils.structarrays import DataFrame
from jafar_yt.profile_dataframe_converter import UserProfileConverter, regions_to_string

# specify countries/categories included in dataset fixture
FIXTURE_COUNTRIES = ['RU']
FIXTURE_CATEGORIES = ['SOCIAL']
# welcome to Omsk!
FIXTURE_CITY = 66
# welcome to Omskaya oblast'!
FIXTURE_REGION = 11318
# welcome to Beregovoy suburb!
FIXTURE_REGIONS = [121538, 108141, 66, 121084, 11318, 59, 225]
FIXTURE_REGION_TYPES = [7, 8, 6, 10, 5, 4, 3, 1, 0]
logger = logging.getLogger(__name__)


class BaseDatasetProcessorMock(object):
    def __init__(self):
        self.path = os.path.join(app.config['BASE_DIR'], 'fixtures')
        self.data = self.load_data()

    def get_data(self, country):
        return DataFrame.from_pandas(self.data)


class AdvisorMongoDatasetProcessorMock(BaseDatasetProcessorMock):
    source = 'advisor_mongo'

    def load_data(self):
        data = pd.read_csv(os.path.join(self.path, 'data.csv'))[['user', 'item', 'value', 'timestamp']]
        data['value'] = data['value'].astype(np.float32)
        return data

    def get_user_features(self, country, features=None):
        users = pd.read_csv(os.path.join(self.path, 'users.csv'))[['user', 'country']]
        df = np.empty(len(users), dtype=UserProfileConverter.get_user_features_dtype())
        df['user'] = users['user']
        # set mock region for locality pipeline
        df['lbs_region_city'] = FIXTURE_CITY
        df['lbs_regions'] = regions_to_string(FIXTURE_REGIONS)

        np.random.seed(8512)
        df['crypta_gender_male'] = np.random.random(len(df))
        df['crypta_age_0_17'] = np.random.random(len(df))
        df['crypta_age_18_24'] = np.random.random(len(df))
        df['crypta_age_25_34'] = np.random.random(len(df))
        df['crypta_age_35_44'] = np.random.random(len(df))
        df['crypta_age_45_99'] = np.random.random(len(df))
        df['crypta_loyalty'] = np.random.random(len(df))
        df['updated_age'] = np.random.random_integers(100, 200, len(df))
        if features is not None:
            df = df[['user'] + list(features)]
        return DataFrame.from_structarray(df)

    def get_item_features(self, country, features=None):
        df = pd.read_csv(os.path.join(self.path, 'items.csv'))[['item', 'category', 'title', 'publisher']]
        df['rating'] = np.random.random(len(df)) * 5
        df['rating_count'] = np.random.random(len(df)) * 500
        with open(os.path.join(self.path, 'json', 'apps.json')) as f:
            descriptions = {app['package_name']: app['description'] for app in json.load(f)}
        df['description'] = np.array(
            [descriptions.get(item) for item in df['item']],
            dtype=np.object
        )
        if features is not None:
            df = df[['item'] + list(features)]
        return DataFrame.from_pandas(df)


class ConversionsDatasetProcessorMock(AdvisorMongoDatasetProcessorMock):
    source = 'conversions'

    def load_data(self):
        data = super(ConversionsDatasetProcessorMock, self).load_data()
        size = 1000
        cr = 0.5
        np.random.seed(1111)
        conversions = pd.DataFrame.from_dict({
            'user': data.user.sample(size, replace=True).values,
            'item': data.item.sample(size, replace=True).values,
            'view_count': np.random.randint(0, 10, size),
            'value': np.random.choice([0, 1], size=size, p=[1.0 - cr, cr]).astype(np.float32)
        })
        return conversions.drop_duplicates()


class ConversionsNoPromoDatasetProcessorMock(ConversionsDatasetProcessorMock):
    source = 'conversions_no_promo'


class LocalityDatasetProcessorMock(BaseDatasetProcessorMock):
    source = 'locality'

    def load_data(self):
        dataset_processor = mock_get_dataset_processor('advisor_mongo')
        df = dataset_processor.get_data(country=FIXTURE_COUNTRIES[0])
        items = np.unique(df['item'])
        np.random.seed(1234)
        n_items = len(items)
        scores = np.arange(n_items) / float(n_items)
        return pd.DataFrame.from_dict(dict(
            region=np.full(n_items, FIXTURE_CITY),
            score=scores,
            item=items
        ))


def mock_get_dataset_processor(dataset_source):
    processors = [
        AdvisorMongoDatasetProcessorMock,
        LocalityDatasetProcessorMock,
        ConversionsDatasetProcessorMock,
        ConversionsNoPromoDatasetProcessorMock,
    ]
    return {p.source: p for p in processors}[dataset_source]()


def dummy_memoize(timeout=None, make_name=None, unless=None):
    def dummy(f):
        return f

    return dummy
