import numpy as np
import os
import pandas as pd
from flask import current_app as app

from jafar.pipelines import ids
from jafar.pipelines.pipeline import Pipeline
from jafar.tests.fixtures.profile import apps as fake_user_apps
from jafar.utils.structarrays import DataFrame


class MockPipeline(Pipeline):
    # can be overriden by pipeline creator
    extra_columns = []

    def __init__(self):
        fixture_path = os.path.join(app.config['BASE_DIR'], 'fixtures')
        self.items = DataFrame.from_pandas(pd.read_csv(os.path.join(fixture_path, 'items.csv'))[['item']])
        self.top_n_count = app.config['TOP_N_COUNT']

    def _get_arbitrary_scores(self, frame):
        # we explicitly avoid using random in tests
        return np.arange(len(frame)) / float(len(frame))

    def predict(self, context):
        """
        Returns the same user-item combinations with arbitrary score.
        """
        result = context.data[ids.FRAME_KEY_TARGET].copy()
        result = result.append_column(self._get_arbitrary_scores(result), 'value')
        return result

    def predict_top_n(self, context):
        basket = context.data.get(ids.FRAME_KEY_BASKET)
        if basket is not None:
            items = self.items[self.items['item'].is_in(basket['item'])]
        else:
            items = self.items.copy()
        items = items[:self.top_n_count]

        users = context.data[ids.FRAME_KEY_TARGET][['user']]
        n_items = len(items)
        n_users = len(users)

        items = items.repeat(n_users)
        users = users.tile(n_items)
        items = items.append_column(users['user'], 'user')
        users = users.append_column(items['item'], 'item')

        predictions = DataFrame.concatenate([users, items])
        predictions = predictions.append_column(self._get_arbitrary_scores(predictions), 'value')

        # add dummy columns to result to mock local/topic/other pipeline behaviour
        for column_spec in self.extra_columns:
            assert 'name' in column_spec and 'values' in column_spec, \
                'Expecting a dict with "name" and "values" fields for a column spec'
            # create a column from `values`:
            n_repeats = int(np.ceil(len(predictions) / float(len(column_spec['values']))))
            value_column = np.repeat(column_spec['values'], n_repeats)[:len(predictions)]
            predictions = predictions.append_column(value_column, column_spec['name'])

        return predictions


class NeighborMockPipeline(MockPipeline):
    """
    Mock for `sonya_neighbors` and `sonya_neighbors_metrika`.
    """

    def __init__(self):
        super(NeighborMockPipeline, self).__init__()
        self.neighbor_dict = {
            item: np.random.choice(self.items['item'], 50)
            for item in fake_user_apps
        }

    def predict(self, context):
        result = []
        for row in context.data[ids.FRAME_KEY_TARGET]:
            neighbors = self.neighbor_dict.get(row['item'])
            if neighbors is not None:
                result.append(DataFrame.from_dict({
                    'user': np.repeat(row['user'], len(neighbors)),
                    'item': neighbors
                }))
        predictions = np.concatenate(result)
        predictions = predictions.append_column(self._get_arbitrary_scores(predictions), 'value')
        return predictions


class PromoMockPipeline(MockPipeline):
    """
    Mock for promo pipeline used in old-fashioned feed recommender, it recommends subset of items.
    """

    def __init__(self):
        super(PromoMockPipeline, self).__init__()
        self.items = self.items[:10]


def get_mock_pipeline_creator(extra_columns=None, mock_pipeline_class=MockPipeline):
    """
    Creates pipeline_creator functions.
    """

    def mock_pipeline_creator(*args, **kwargs):
        pipeline = mock_pipeline_class()
        pipeline.top_n_count = kwargs.get('top_n', app.config['TOP_N_COUNT'])
        pipeline.extra_columns = extra_columns or []
        return pipeline

    return mock_pipeline_creator
