import types
from uuid import UUID

import mock

from jafar import advisor_mongo
from jafar.feed import recommenders
from jafar.feed.base import FeedBlock
from jafar.models.cards import Card
from jafar.tests import JafarTestCase
from jafar.tests.fixtures.profile import apps as fake_user_apps, fake_user_profile, user_id as fake_user_id
from jafar.tests.mocks.datasets import FIXTURE_CITY
from jafar.tests.mocks.pipeline import get_mock_pipeline_creator, NeighborMockPipeline, PromoMockPipeline


class FeedRecommenderTestCaseMixin(object):
    extra_columns = []

    def setUp(self):
        super(FeedRecommenderTestCaseMixin, self).setUp()
        fake_user_profile()
        self.mock_feed = self.get_feed_with_filter()

    def tearDown(self):
        advisor_mongo.db.profile.drop()
        super(FeedRecommenderTestCaseMixin, self).tearDown()

    def get_recommender_instance(self):
        raise NotImplementedError

    def get_two_apps_card(self):
        return Card(
            min_count=2, reserved_count=1, max_count=2,
            card_type='Singleapp_card_Expandable', supported_content_types=('apps', 'promo')
        )

    def get_small_card(self, item_count, reserved_count=None):
        if not reserved_count:
            reserved_count = item_count - 1
        return Card(
            min_count=item_count, max_count=item_count, reserved_count=reserved_count,
            card_type='Multiapps_card_multirows', supported_content_types=('apps', 'promo')
        )

    def get_extra_large_card(self):
        return Card(
            min_count=9000, reserved_count=8999,
            card_type='Scrollable_Expandable', supported_content_types=('apps', 'promo')
        )

    def get_recommendation_group_shortcut(self):
        recommender = self.get_recommender_instance()
        recommendation_groups = recommender.get_recommendation_groups(self.mock_feed, mock.Mock())
        return recommendation_groups[0]

    def simple_get_block_shortcut(self):
        recommendation_group = self.get_recommendation_group_shortcut()
        card = self.get_two_apps_card()
        return recommendation_group.get_blocks([card])[0]

    def test_get_recommendation_group(self):
        recommendation_group = self.get_recommendation_group_shortcut()
        # not testing for an instance of a specific class: recommendation group
        # has to have 'size' property and a few additional methods
        self.assertTrue(hasattr(recommendation_group, 'size'))
        self.assertTrue(recommendation_group.size > 0)
        self.assertTrue(hasattr(recommendation_group, 'get_blocks'))
        self.assertTrue(hasattr(recommendation_group, 'filter'))
        self.assertTrue(hasattr(recommendation_group, 'update_filter'))

    def test_simple_get_block(self):
        block = self.simple_get_block_shortcut()
        card = self.get_two_apps_card()
        self.assertIsInstance(block, FeedBlock)
        self.assertEquals(len(block.items), 2)
        self.assertEquals(len(block.reserved), 1)
        for lst in (block.items, block.reserved):
            for item in lst:
                self.assertIn('score', item)
                self.assertIn('package_name', item)
        self.assertEquals(block.card_type, card.card_type)
        self.assertEquals(block.title, self.title)
        self.assertEquals(block.subtitle, self.subtitle)
        self.assertEquals(block.rotation_interval, card.rotation_interval)

    def test_one_time_item_filtering(self):
        """
        Two consecutive blocks shouldn't contain intersections.
        """
        recommendation_group = self.get_recommendation_group_shortcut()
        card1 = self.get_small_card(item_count=3)
        card2 = self.get_small_card(item_count=4)
        # one preload, two requests
        block1, block2 = recommendation_group.get_blocks([card1, card2])
        items1 = [item['package_name'] for item in block1.items]
        items2 = [item['package_name'] for item in block2.items]

        self.assertEquals(len(set(items1).intersection(set(items2))), 0)

    def get_feed_with_filter(self):
        feed = mock.MagicMock()
        feed.item_filter = set()

        def update_meta(self, block):
            for item in block.items:
                self.item_filter.add(item['package_name'])

        feed.update_meta = types.MethodType(update_meta, feed)
        feed.user.clids = {}
        return feed


promo_extra_columns = [
    {'name': 'offer_id', 'values': ['appnext_12345', 'cheetah_54321']},
    {'name': 'cpm', 'values': [1.1, 0.9]}
]

local_extra_columns = [{'name': 'lbs_region_city', 'values': [FIXTURE_CITY]}]


class SimpleLocalFeedRecommenderTestCase(FeedRecommenderTestCaseMixin, JafarTestCase):
    title = 'local_apps'
    subtitle = 'local_apps_subtitle'
    extra_columns = local_extra_columns

    def get_recommender_instance(self):
        recommender = recommenders.SimpleLocalFeedRecommender(title_pairs=[
            {'title': self.title, 'subtitle': self.subtitle}
        ])
        recommender.pipeline_creator = get_mock_pipeline_creator(extra_columns=local_extra_columns)
        return recommender

    def test_explanation(self):
        block = self.simple_get_block_shortcut()
        self.assertEquals(block.explanation, {'region_id': FIXTURE_CITY})


# NOTE: these feed recommenders are parametrized by pipeline
# so no need to patch pipeline creator for test case.


similar_extra_columns = [{'name': 'similar_to', 'values': fake_user_apps}]


class SimilarItemsRecommenderTestCase(FeedRecommenderTestCaseMixin, JafarTestCase):
    title = 'similar_apps'
    subtitle = 'similar_apps_subtitle'

    def get_recommender_instance(self):
        recommender = recommenders.SimilarItemsRecommender(pipeline='sonya_neighbors', title_pairs=[
            {'title': self.title, 'subtitle': self.subtitle}
        ])
        recommender.pipeline_creator = get_mock_pipeline_creator(mock_pipeline_class=NeighborMockPipeline,
                                                                 extra_columns=similar_extra_columns)
        return recommender

    def test_explanations(self):
        """
        SimilarItemsRecommender should return user's items as explanations.
        """
        recommendation_group = self.get_recommendation_group_shortcut()
        card = self.get_two_apps_card()
        for _ in xrange(3):
            block = recommendation_group.get_blocks([card])[0]
            self.assertIn('similar_to', block.explanation)
            explanation = block.explanation['similar_to']
            self.assertIn(explanation, fake_user_apps)


class TopNRecommenderTestCase(FeedRecommenderTestCaseMixin, JafarTestCase):
    title = 'recommended_apps'
    subtitle = 'recommended_apps_subtitle'

    def get_recommender_instance(self):
        recommender = recommenders.TopNRecommender(pipeline='kano', title_pairs=[
            {'title': self.title, 'subtitle': self.subtitle}
        ])
        recommender.pipeline_creator = get_mock_pipeline_creator()
        return recommender

    def test_explanation(self):
        block = self.simple_get_block_shortcut()
        self.assertEquals(block.explanation, None)


class TopNCategorizedRecommenderTestCase(FeedRecommenderTestCaseMixin, JafarTestCase):
    title = 'recommended_in_category'
    subtitle = 'recommended_in_category_subtitle'

    def add_user_to_feed(self, feed):
        """
        Adds user's device_id to mock feed so TopNCategorizedRecommender can
        fetch used categories from user's usage stats.
        """
        feed.user.device_id = UUID(fake_user_id)
        return feed

    def get_feed_with_filter(self):
        feed = super(TopNCategorizedRecommenderTestCase, self).get_feed_with_filter()
        return self.add_user_to_feed(feed)

    def get_recommender_instance(self):
        recommender = recommenders.TopNCategorizedRecommender(pipeline='kano', title_pairs=[
            {'title': self.title, 'subtitle': self.subtitle}
        ])
        recommender.pipeline_creator = get_mock_pipeline_creator(
            extra_columns=[{'name': 'category', 'values': ['COMMUNICATION', 'WEATHER', 'GAME_ACTION']}]
        )
        return recommender
