# coding=utf-8
from unittest import TestCase

from jafar.image_composer import ImageComposer
from jafar.tests.fixtures import get_fixture_path
from PIL import Image, ImageChops


class ImageComposerTestCase(TestCase):
    density = 420

    def setUp(self):
        super(ImageComposerTestCase, self).setUp()

    def _compose_test_image(self, img_path, full_size):
        composer = ImageComposer(get_fixture_path(img_path), '#264399', self.density)
        composer.set_full_size(full_size)
        composer.set_title("Bla bla bla", '#ff0000')
        composer.set_subtitle("Abra kadabra", '#ff0000')

        return composer.compose()

    def test_no_title_no_gradient(self):
        image = ImageComposer(
            get_fixture_path('images/cov_sport.jpg'),
            '#264399',
            self.density
        ).compose()
        self.assertIsNotNone(image)
        # TODO: create a picture for comparison
        # image.save(os.path.join(BASE_DIR, 'cover_no_title_no_gradient.png'))

    def test_no_title_with_gradient(self):
        composer = ImageComposer(
            get_fixture_path('images/cov_sport.jpg'),
            '#264399',
            self.density
        )
        composer.add_gradient()
        image = composer.compose()
        self.assertIsNotNone(image)
        # TODO: create a picture for comparison
        # image.save(os.path.join(BASE_DIR, 'cover_no_title_with_gradient.png'))

    def test_one_title_no_gradient_hex_color(self):
        composer = ImageComposer(
            get_fixture_path('images/cov_sport.jpg'),
            '#264399',
            self.density
        )
        composer.set_title('Sportsportsportsportsportsportsportsportsportsports', '#FFFFFF')
        image = composer.compose()
        self.assertIsNotNone(image)
        # TODO: create a picture for comparison
        # image.save(os.path.join(BASE_DIR, 'cover_one_title_no_gradient_hex_color.png'))

    def test_one_title_with_gradient_hex_color(self):
        composer = ImageComposer(
            get_fixture_path('images/cov_sport.jpg'),
            '#264399',
            self.density
        )
        composer.set_title('Sports', '#FFFFFF')
        composer.add_gradient()
        image = composer.compose()
        self.assertIsNotNone(image)
        # TODO: create a picture for comparison
        # image.save(os.path.join(BASE_DIR, 'cover_one_title_with_gradient_hex_color.png'))

    def test_two_titles_no_gradient_rgb_color(self):
        composer = ImageComposer(
            get_fixture_path('images/cov_animals.jpg'),
            (14, 14, 14),
            self.density
        )
        composer.set_title('Sports', '#FFFFFF')
        composer.set_subtitle('by Yandex.Sport team', '#FFFFFF')
        image = composer.compose()
        self.assertIsNotNone(image)
        # TODO: create a picture for comparison
        # image.save(os.path.join(BASE_DIR, 'cover_two_titles_no_gradient_rgb_color.png'))

    def test_two_titles_with_gradient_rgb_color(self):
        composer = ImageComposer(
            get_fixture_path('images/cov_animals.jpg'),
            (14, 14, 14),
            self.density
        )
        composer.set_title('Sports', '#FFFFFF')
        composer.set_subtitle('by Yandex.Sport team', '#FFFFFF')
        composer.add_gradient()
        image = composer.compose()
        self.assertIsNotNone(image)
        # TODO: create a picture for comparison
        # image.save(os.path.join(BASE_DIR, 'cover_two_titles_with_gradient_rgb_color.png'))

    def test_hex_to_rgb__should_convert(self):
        self.assertEquals(ImageComposer.hex_to_rgb('#FFFFFF'), (255, 255, 255))
        self.assertEquals(ImageComposer.hex_to_rgb('aaaaaa'), (170, 170, 170))
        self.assertEquals(ImageComposer.hex_to_rgb('#FF000000'), (0, 0, 0, 255))
        self.assertEquals(ImageComposer.hex_to_rgb((180, 180, 180)), (180, 180, 180))

    def test_hex_to_rgb__should_fail(self):
        self.assertRaises(ValueError, ImageComposer.hex_to_rgb, None)
        self.assertRaises(ValueError, ImageComposer.hex_to_rgb, {})
        self.assertRaises(ValueError, ImageComposer.hex_to_rgb, '#abcdefjh')
        self.assertRaises(ValueError, ImageComposer.hex_to_rgb, '#GGGGGG')
        self.assertRaises(ValueError, ImageComposer.hex_to_rgb, (256, 256, 256))
        self.assertRaises(ValueError, ImageComposer.hex_to_rgb, (-1, -2, -3))

    def test_ellipsize(self):
        self.assertEquals(ImageComposer.ellipsize('Super text for tests!', 10), 'Super text...')
        self.assertEquals(ImageComposer.ellipsize('Super text', 40), 'Super text')

    def test_create_gradient(self):
        # TODO: create a picture for comparison
        self.assertIsNotNone(ImageComposer.create_gradient((100, 100, 100), (5, 5)))

    def test_not_full_size_image_created(self):
        composer = ImageComposer(
            get_fixture_path('images/cov_sport.jpg'),
            '#264399',
            self.density
        )

        composer.set_full_size(False)

        self.assertIsNotNone(composer.compose())

    def test_full_size_image_created(self):
        composer = ImageComposer(
            get_fixture_path('images/cov_sport.jpg'),
            '#264399',
            self.density
        )

        composer.set_full_size(True)

        self.assertIsNotNone(composer.compose())

    def test_square_img_no_full_size(self):
        actual_result = self._compose_test_image(get_fixture_path('images/cov_sport.jpg'), False)

        diff = ImageChops.difference(actual_result,
                                     Image.open(get_fixture_path('images/results/square_img_not_full_size.png')))
        assert diff.convert("L").getextrema() == (0, 0)

    def test_square_img_full_size(self):
        actual_result = self._compose_test_image(get_fixture_path('images/cov_sport.jpg'), True)

        diff = ImageChops.difference(actual_result,
                                     Image.open(get_fixture_path('images/results/square_img_full_size.png')))
        assert diff.convert("L").getextrema() == (0, 0)

    def test_panorama_not_img_full_size(self):
        actual_result = self._compose_test_image(get_fixture_path('images/panorama_image.jpg'), False)

        diff = ImageChops.difference(actual_result,
                                     Image.open(get_fixture_path('images/results/panorama_img_not_full_size.png')))
        assert diff.convert("L").getextrema() == (0, 0)

    def test_panorama_img_full_size(self):
        actual_result = self._compose_test_image(get_fixture_path('images/panorama_image.jpg'), True)

        diff = ImageChops.difference(actual_result,
                                     Image.open(get_fixture_path('images/results/panorama_img_full_size.png')))
        assert diff.convert("L").getextrema() == (0, 0)
