import numpy as np

from jafar.pipelines import Pipeline, ids
from jafar.pipelines.blocks.features import ItemFeaturesBlock, ItemsStoreBlock
from jafar.pipelines.blocks.mapping import MappingBlock
from jafar.tests.unittests.pipelines.blocks import BlockTestCase
from jafar.utils.structarrays import DataFrame


class GooglePlayFeaturesBlockTestCase(BlockTestCase):
    def test_google_play_features_blocks(self):
        apps = [
            (4.030572891, 'com.gau.go.touchhelperex.theme.technology', 7294),
            (2.5, 'org.sysprocess.xmonitor', 2),
            (4.478260994, 'com.torino.today', 46),
            (4.0, 'mobi.mpk.kursk', 194),
            (4.224669456, 'ru.chefe.February23', 227),
            (3.13269496, 'com.google.vr.vrcore', 2193),
        ]

        item_features = DataFrame(np.array(apps, dtype=[('rating', np.float32),
                                                        ('item', np.object),
                                                        ('rating_count', np.uint32)]))
        user = 'a624d5e5-b136-4296-8c63-996a0cf22309'
        predictions = DataFrame.from_structarray(np.array(
            [(user, item, 1.0) for item in item_features['item']],
            dtype=[('user', np.object), ('item', np.object), ('value', np.float32)]
        ))
        store_block = ItemsStoreBlock(input_frame=ids.FRAME_KEY_ITEM_FEATURES,
                                      feature_names=['rating', 'rating_count'])
        feature_block = ItemFeaturesBlock(input_frame=ids.FRAME_KEY_PREDICTIONS,
                                          feature_names=['rating', 'rating_count'])
        pipeline = Pipeline(name='gp_test', storage=self.memory_storage, blocks=[
            ('mapping', MappingBlock(nested_blocks=[('store_block', store_block),
                                                    ('feature_block', feature_block)],
                                     frames=[ids.FRAME_KEY_ITEM_FEATURES, ids.FRAME_KEY_PREDICTIONS])),
        ])

        # train and predict modes are the same
        for train in (True, False):
            context = self.get_test_context(pipeline)
            context.data[ids.FRAME_KEY_PREDICTIONS] = predictions
            context.data[ids.FRAME_KEY_ITEM_FEATURES] = item_features
            result_context = pipeline.apply_blocks(train, context)

            self.assert_same_context(result_context, context)
            self.assertIn(ids.FRAME_KEY_PREDICTIONS, context.data)
            if train:
                self.assertIn(ids.FRAME_KEY_ITEM_FEATURES, context.data)

            for feature in ['rating', 'rating_count']:
                value = item_features[feature]
                self.assertIn(feature, result_context.data[ids.FRAME_KEY_PREDICTIONS].dtype.names)
                self.assertTrue(np.allclose(result_context.data[ids.FRAME_KEY_PREDICTIONS][feature], value))
            # also verify predictions frame isn't damaged
            for field in ('user', 'item', 'value'):
                self.assertIn(field, result_context.data[ids.FRAME_KEY_PREDICTIONS].dtype.names)
