import datetime
import logging
import threading
from time import time

from flask import g, current_app as app
from werkzeug.contrib.cache import SimpleCache

logger = logging.getLogger(__name__)


class FastSimpleCache(SimpleCache):
    """Fast Simple memory cache is the same as SimpleCache exclude (un)pickling values

    :param threshold: the maximum number of items the cache stores before
                      it starts deleting some.
    :param default_timeout: the default timeout that is used if no timeout is
                            specified on :meth:`~BaseCache.set`. A timeout of
                            0 indicates that the cache never expires.
    """

    def get(self, key):
        try:
            expires, value = self._cache[key]
            if expires == 0 or expires > time():
                return value
        except KeyError:
            return None

    def set(self, key, value, timeout=None):
        expires = self._normalize_timeout(timeout)
        self._prune()
        self._cache[key] = (expires, value)
        return True

    def add(self, key, value, timeout=None):
        expires = self._normalize_timeout(timeout)
        self._prune()
        item = (expires, value)
        if key in self._cache:
            return False
        self._cache.setdefault(key, item)
        return True


def fast_simple(app, config, args, kwargs):
    kwargs.update(dict(threshold=config['CACHE_THRESHOLD']))
    return FastSimpleCache(*args, **kwargs)


class RequestScopeCache(SimpleCache):
    """
    Request scope cache that stores variables in flask.g
    """

    def get(self, key):
        return getattr(g, key, None)

    def set(self, key, value, timeout=None):
        setattr(g, key, value)
        return True

    def add(self, key, value, timeout=None):
        if key in g:
            return False
        setattr(g, key, value)
        return True


def request_cache(app, config, args, kwargs):
    return RequestScopeCache(*args, **kwargs)


class ObjectCache(object):
    """
    Object cache initializes its update every expiration_interval.
    It starts separate thread when cache is expired and load fresh data.


    You should implement the load() method and any convenient data getters.
    get() in consumers returns the same object as load()
    """
    expiration_interval = datetime.timedelta(hours=24)

    def __init__(self):
        self._updated_at = None
        self._lock = threading.BoundedSemaphore()

    def force_reload(self):
        """
        Force the reload of cache synchronously. Used in tests.
        """
        with app.app_context():
            self._data = self.load()
            self._updated_at = datetime.datetime.utcnow()

    def _load_with_context(self, context):
        """
        Used to call self.load in separate thread. Pass flask context here.
        """
        if self._lock.acquire(blocking=False):
            # noinspection PyBroadException
            try:
                with context:
                    logger.debug('Start loading %s', self.__class__.__name__)
                    self._data = self.load()
                    logger.debug('Finish loading %s', self.__class__.__name__)
                self._updated_at = datetime.datetime.utcnow()
            except:
                logger.exception('%s loading exception', self.__class__.__name__)
            finally:
                self._lock.release()

    def get(self):
        self._check_cache()
        return self._data

    def _check_cache(self):
        if not self.is_ready() or self.is_expired():
            updating_thread = threading.Thread(target=self._load_with_context,
                                               kwargs=dict(context=app.app_context()))
            updating_thread.start()

    def is_ready(self):
        return self._updated_at is not None

    def is_expired(self):
        return datetime.datetime.utcnow() - self._updated_at >= self.expiration_interval

    def load(self):
        raise NotImplementedError
