import logging
import time
from datetime import date, timedelta
from operator import itemgetter

from flask import current_app as app
from yt import wrapper as yt
from yt.transfer_manager import client as tm

logger = logging.getLogger(__name__)


def mount_dynamic_table(dyn_table, yt_client, symlink):
    current_dyn_table = get_symlink_target(symlink, yt_client) if yt_client.exists(symlink) else ''
    if current_dyn_table == dyn_table:
        logger.info('Dynamic table was mounted. Exiting.')
        return

    logger.info('Mounting new dynamic table %s', dyn_table)
    yt_client.mount_table(dyn_table, freeze=True, sync=True)
    wait_for_preload(dyn_table, yt_client)
    logger.info('Changing symlink target from %s to %s', current_dyn_table, dyn_table)
    yt_client.link(dyn_table, symlink, force=True)
    logger.info('Waiting 5 minutes for YT caches update')
    time.sleep(5 * 60)
    if current_dyn_table:
        logger.info('Unmounting old dynamic table %s', current_dyn_table)
        yt_client.unmount_table(current_dyn_table, sync=True)


def get_symlink_target(symlink, yt_client):
    return yt_client.get_attribute(symlink + '&', 'target_path')


def wait_for_preload(dyn_table, yt_client):
    def get_stats():
        return yt_client.get_attribute(dyn_table, 'tablet_statistics')

    def is_preloaded():
        stats = get_stats()
        return stats['preload_pending_store_count'] == 0 and stats['preload_failed_store_count'] == 0

    start_time = time.time()
    while not is_preloaded():
        if time.time() - start_time > TABLE_PRELOAD_TIMEOUT:
            raise Exception('Table preload timeout exceeded')
        if get_stats()['preload_failed_store_count'] > 0:
            raise Exception('Table preload failed')
        time.sleep(1)
    logger.info('Table %s is preloaded', dyn_table)


TABLE_MOUNT_TIMEOUT = 10 * 60
TABLE_PRELOAD_TIMEOUT = 5 * 60
TABLE_COPY_TIMEOUT = 60 * 60


def cleanup_old_tables(base_path, ttl, yt_client, exclude=None):
    exclude = [] if exclude is None else exclude
    expiration_date = date.today() - timedelta(days=ttl)
    for table in yt_client.list(base_path):
        path = yt.ypath_join(base_path, table)
        if yt_client.get(path, attributes=['tablet_state']).attributes.get('tablet_state') == 'frozen':
            logger.warning('Table %s if mounted in state "frozen", skip', path)
            continue

        if table < expiration_date.isoformat() and path not in exclude:
            logger.info('Deleting expired table %s', path)
            yt_client.remove(path)


def dynamize_static_and_cleanup(table, schema, cleanup_kwargs_list, in_memory_mode, mount=False, symlink=None):
    """
    Takes static YT table on primary YT cluster, makes it dynamic, mounts it and cleanup old ones
    :param table: prepared static table
    :param schema: yt.yson schema
    :param symlink: dynamic table symlink
    :param cleanup_kwargs_list: list of kwargs to cleanup_old_tables func
    :param mount: whether to mount dyn table
    :param in_memory_mode: in_memory_mode for dyn table
    :return:
    """
    copy_table_to_reserve_cluster(table, schema=schema, in_memory_mode=in_memory_mode)
    for cluster in itemgetter('YT_PROXY', 'YT_RESERVE_PROXY')(app.config):
        yt_client = yt.YtClient(cluster, app.config['YT_TOKEN'])
        logger.debug('Making table dynamic')
        if not yt_client.get(table, attributes=['dynamic']).attributes['dynamic']:
            yt_client.alter_table(table, dynamic=True)
        if mount:
            mount_dynamic_table(table, yt_client, symlink)
        current_table = get_symlink_target(app.config['YT_PATH_DYNAMIC_LINK'], yt_client)
        for kwargs in cleanup_kwargs_list:
            kwargs['yt_client'] = yt_client
            kwargs['exclude'] = [current_table]
            cleanup_old_tables(**kwargs)


def copy_table_to_reserve_cluster(table, schema, in_memory_mode):
    yt_client = yt.YtClient(proxy=app.config['YT_RESERVE_PROXY'], token=app.config['YT_TOKEN'])
    if yt_client.exists(table):
        return

    create_dyntable(table, yt_client, schema=schema, in_memory_mode=in_memory_mode)
    logger.info('Copying %s to %s', table, app.config['YT_RESERVE_PROXY'])
    tm.add_task(
        source_cluster=app.config['YT_PROXY'].split('.')[0],
        destination_cluster=app.config['YT_RESERVE_PROXY'].split('.')[0],
        source_table=table,
        destination_table=table,
        sync=True,  # Wait until done
        params=dict(
            pool=app.config['YT_POOL'],
        ))


def create_dyntable(result_table, yt_client, schema, in_memory_mode):
    yt_client.create_table(
        path=result_table,
        attributes=dict(
            schema=schema,
            tablet_cell_bundle=app.config['YT_TABLET_CELL_BUNDLE'],
            in_memory_mode=in_memory_mode,
            dynamic=False,
        ))
