cimport cython
from cython.parallel import prange, parallel
from libc.math cimport log2

cdef inline int int_min(int a, int b) nogil: return a if a <= b else b
cdef inline float float_min(float a, float b) nogil: return a if a <= b else b


@cython.cdivision(True)
@cython.boundscheck(False)
@cython.wraparound(False)
cpdef float roc_auc_score_user(float [:] y_true_user, float [:] y_score_user) nogil:
    cdef float correct, total, auc
    cdef unsigned int N_user_items, j, k

    N_user_items = y_true_user.shape[0]
    correct = 0
    total = 0
    for j in range(N_user_items):
        for k in range(N_user_items):
            if (y_true_user[j] == y_true_user[k]):
                continue
            total += 1
            if y_score_user[j] == y_score_user[k]:
                correct += 0.5
            elif (y_true_user[j] < y_true_user[k]) and (y_score_user[j] < y_score_user[k]):
                correct += 1
            elif (y_true_user[j] > y_true_user[k]) and (y_score_user[j] > y_score_user[k]):
                correct += 1
    if correct == 0:
        return 0.
    else:
        return correct / total


@cython.cdivision(True)
@cython.boundscheck(False)
@cython.wraparound(False)
cpdef float roc_auc_score_avg(float [:] y_true, float [:] y_score, int [:, :] indices):
    cdef unsigned int i
    cdef int N_users
    cdef float result

    N_users = indices.shape[0]
    result = 0

    with nogil, parallel():
        for i in prange(N_users, schedule='guided'):
            result += roc_auc_score_user(
                y_true[indices[i][0]: indices[i][1]],
                y_score[indices[i][0]: indices[i][1]]
            )

    result = result / N_users
    return result


@cython.cdivision(True)
@cython.boundscheck(False)
@cython.wraparound(False)
cpdef float unexpectedness_user(int [:] expected, int [:] predicted) nogil:
    cdef int i, j, N, M
    cdef float expected_count
    N = expected.shape[0]
    M = predicted.shape[0]
    expected_count = 0
    for i in range(M):
        for j in range(N):
            if predicted[i] == expected[j]:
                expected_count += 1
                break
    if int_min(N, M) == 0:
        return 0
    else:
        return float_min((M - expected_count), N) / int_min(N, M)



@cython.cdivision(True)
@cython.boundscheck(False)
@cython.wraparound(False)
cpdef float unexpectedness_score(int [:] expected, int [:, :] excepted_indices,
                                 int [:] predicted, int[:, :] predicted_indices):
    cdef unsigned int i
    cdef int N_users
    cdef float result

    N_users = excepted_indices.shape[0]
    result = 0

    with nogil, parallel():
        for i in prange(N_users, schedule='guided'):
            result += unexpectedness_user(
                expected[excepted_indices[i][0]: excepted_indices[i][1]],
                predicted[predicted_indices[i][0]: predicted_indices[i][1]]
            )

    result = result / N_users
    return result


@cython.boundscheck(False)
@cython.wraparound(False)
cpdef int[:] binarize_predicted(int[:] actual, int[:] predicted) nogil:
    cdef int i, j, N, M, found

    # binarize predicted array to relevant/irrelevant scores
    # example:
    #   actual = [1, 2, 3, 4]
    #   predicted = [7, 2, 8, 1]
    #   binary_predicted = [0, 1, 0, 1]

    N = actual.shape[0]
    M = predicted.shape[0]
    for i in range(M):
        found = 0
        for j in range(N):
            if predicted[i] == actual[j]:
                predicted[i] = 1
                found = 1
                break
        if found == 0:
            predicted[i] = 0
    return predicted


@cython.cdivision(True)
@cython.boundscheck(False)
@cython.wraparound(False)
cpdef float map_user(int [:] actual, int [:] predicted, int limit_top_n) nogil:
    cdef float score, precision_at_k
    cdef int i, k, N, M

    predicted = binarize_predicted(actual, predicted)
    score = 0.0
    N = int_min(predicted.shape[0], limit_top_n)
    for k in range(N):
        if predicted[k] != 0:
            precision_at_k = 0.0
            for i in range(k + 1):
                precision_at_k += predicted[i]
            precision_at_k /= (k + 1)

            score += precision_at_k
    if N == 0:
        return 0
    else:
        return score / N


@cython.cdivision(True)
@cython.boundscheck(False)
@cython.wraparound(False)
cpdef float map_score(int [:] actual_items, int [:, :] actual_indices,
                      int [:] predicted_items, int[:, :] predicted_indices,
                      int limit_top_n):
    cdef unsigned int i
    cdef int N_users
    cdef float result

    N_users = actual_indices.shape[0]
    result = 0

    with nogil, parallel():
        for i in prange(N_users, schedule='guided'):
            result += map_user(
                actual_items[actual_indices[i][0]: actual_indices[i][1]],
                predicted_items[predicted_indices[i][0]: predicted_indices[i][1]],
                limit_top_n
            )

    result = result / N_users
    return result



@cython.cdivision(True)
@cython.boundscheck(False)
@cython.wraparound(False)
cpdef float ndcg_user(int [:] actual, int [:] predicted, int limit_top_n) nogil:
    cdef float real_score, ideal_score
    cdef int i, j, N, hits, found

    predicted = binarize_predicted(actual, predicted)
    N = int_min(predicted.shape[0], limit_top_n)

    # then discount relevance by position and sum
    real_score = 0.0
    for i in range(N):
        real_score += predicted[i] / log2(i + 2)  # why 2?

    # count number of hits for max possible ndcg
    hits = 0
    for i in range(N):
        hits += predicted[i]

    # then compute ideal dcg (for the case where all relevant items
    # are located at the beginning of the predicted list)
    for i in range(hits):
        predicted[i] = 1
    for i in range(N - hits):
        predicted[i + hits] = 0

    ideal_score = 0.0
    for i in range(N):
        ideal_score += predicted[i] / log2(i + 2)  # why 2?

    if ideal_score == 0:
        return 0.0
    else:
        return real_score / ideal_score


@cython.cdivision(True)
@cython.boundscheck(False)
@cython.wraparound(False)
cpdef float ndcg_score(int [:] actual_items, int [:, :] actual_indices,
                       int [:] predicted_items, int[:, :] predicted_indices,
                       int limit_top_n):
    cdef unsigned int i
    cdef int N_users
    cdef float result

    N_users = actual_indices.shape[0]
    result = 0

    with nogil, parallel():
        for i in prange(N_users, schedule='guided'):
            result += ndcg_user(
                actual_items[actual_indices[i][0]: actual_indices[i][1]],
                predicted_items[predicted_indices[i][0]: predicted_indices[i][1]],
                limit_top_n
            )

    result = result / N_users
    return result


@cython.cdivision(True)
@cython.boundscheck(False)
@cython.wraparound(False)
cpdef float mrr_user(int [:] actual, int [:] predicted, int limit_top_n) nogil:
    cdef float score
    cdef int i, j, N, M, found

    score = 0.0
    found = 0
    N = actual.shape[0]
    M = int_min(predicted.shape[0], limit_top_n)
    for i in range(M):
        if found == 1:
            break
        for j in range(N):
            if predicted[i] == actual[j]:
                score = 1. / (i + 1)
                found = 1
                break
    return score


@cython.cdivision(True)
@cython.boundscheck(False)
@cython.wraparound(False)
cpdef float mrr_score(int [:] actual_items, int [:, :] actual_indices,
                      int [:] predicted_items, int[:, :] predicted_indices,
                      int limit_top_n):
    cdef unsigned int i
    cdef int N_users
    cdef float result

    N_users = actual_indices.shape[0]
    result = 0

    with nogil, parallel():
        for i in prange(N_users, schedule='guided'):
            result += mrr_user(
                actual_items[actual_indices[i][0]: actual_indices[i][1]],
                predicted_items[predicted_indices[i][0]: predicted_indices[i][1]],
                limit_top_n
            )

    result = result / N_users
    return result


@cython.cdivision(True)
@cython.boundscheck(False)
@cython.wraparound(False)
cpdef float precision_user(int [:] actual, int [:] predicted, int limit_top_n) nogil:
    cdef float score
    cdef int N, i

    predicted = binarize_predicted(actual, predicted)
    N = int_min(predicted.shape[0], limit_top_n)
    score = 0.0
    for i in xrange(N):
        score += predicted[i]

    if N == 0:
        return 0
    else:
        return score / N


@cython.cdivision(True)
@cython.boundscheck(False)
@cython.wraparound(False)
cpdef float precision_score(int [:] actual_items, int [:, :] actual_indices,
                            int [:] predicted_items, int[:, :] predicted_indices,
                            int limit_top_n):
    cdef unsigned int i
    cdef int N_users
    cdef float result

    N_users = actual_indices.shape[0]
    result = 0

    with nogil, parallel():
        for i in prange(N_users, schedule='guided'):
            result += precision_user(
                actual_items[actual_indices[i][0]: actual_indices[i][1]],
                predicted_items[predicted_indices[i][0]: predicted_indices[i][1]],
                limit_top_n
            )

    result = result / N_users
    return result


@cython.cdivision(True)
@cython.boundscheck(False)
@cython.wraparound(False)
cpdef float recall_user(int [:] actual, int [:] predicted, int limit_top_n) nogil:
    cdef float score
    cdef int N, i

    predicted = binarize_predicted(actual, predicted)
    N = int_min(predicted.shape[0], limit_top_n)
    score = 0.0
    for i in xrange(N):
        score += predicted[i]

    N = actual.shape[0]
    if N == 0:
        return 0
    else:
        return score / N


@cython.cdivision(True)
@cython.boundscheck(False)
@cython.wraparound(False)
cpdef float recall_score(int [:] actual_items, int [:, :] actual_indices,
                            int [:] predicted_items, int[:, :] predicted_indices,
                            int limit_top_n):
    cdef unsigned int i
    cdef int N_users
    cdef float result

    N_users = actual_indices.shape[0]
    result = 0

    with nogil, parallel():
        for i in prange(N_users, schedule='guided'):
            result += recall_user(
                actual_items[actual_indices[i][0]: actual_indices[i][1]],
                predicted_items[predicted_indices[i][0]: predicted_indices[i][1]],
                limit_top_n
            )

    result = result / N_users
    return result

