import requests
from flask import current_app as app


class HasOffersAPIError(Exception):
    pass


class HasOffersAPI(object):

    def __init__(self):
        self.session = requests.session()

    def _request_hasoffers_api(self, params):
        try:
            url = app.config['HASOFFERS_URL'].format(app.config['HASOFFERS_API_TOKEN'])
            response = self.session.get(url, params=params, allow_redirects=False)
            response.raise_for_status()

            content = response.json()['response']

            error_masage = content.get('errorMessage')
            if error_masage:
                raise HasOffersAPIError(error_masage)

            return content['data']

        except requests.RequestException as e:
            raise HasOffersAPIError(e)

    def get_approved_affiliate_ids(self, offer_id):
        params = {
            'Target': 'Offer',
            'Method': 'getApprovedAffiliateIds',
            'id': offer_id,
        }
        data = self._request_hasoffers_api(params)
        return data

    def get_target_countries(self, offer_id):
        params = {
            'Target': 'Offer',
            'Method': 'getTargetCountries',
            'id': offer_id,
        }
        data = self._request_hasoffers_api(params)
        if data is not None:
            return [item['Country']['code'] for item in data]
        return data

    def get_offer(self, offer_id):
        params = {
            'Target': 'Offer',
            'Method': 'findById',
            'id': offer_id,
        }
        data = self._request_hasoffers_api(params)
        return data

    def get_affiliate(self, affiliate_id):
        params = {
            'Target': 'Affiliate',
            'Method': 'findById',
            'id': affiliate_id,
        }
        data = self._request_hasoffers_api(params)
        return data

    def get_affiliate_postback_urls(self, affiliate_id):
        params = {
            'Target': 'Affiliate',
            'Method': 'getOfferPixels',
            'id': affiliate_id,
            'status': 'active',
        }
        data = self._request_hasoffers_api(params)
        postbacks = []
        if data:
            for item in data.itervalues():
                postbacks.append(item['OfferPixel']['code'])
        return postbacks

    def get_affiliate_api_keys_and_users(self):
        params = {
            'Target': 'ApiKey',
            'Method': 'findApiKeys',
            'filters[user_type]': 'affiliate',
            'filters[status]': 'active',
        }
        data = self._request_hasoffers_api(params)
        api_keys_by_users = {}
        for item in data.values():
            api_keys_by_users[item['user_id']] = item['api_key']
        return api_keys_by_users

    def get_users_by_affiliate_id(self, affiliate_id):
        params = {
            'Target': 'AffiliateUser',
            'Method': 'findAllIdsByAffiliateId',
            'affiliate_id': affiliate_id,
        }
        data = self._request_hasoffers_api(params)
        return data
