import time
import functools
import logging


logger = logging.getLogger(__name__)


def safe_memoize(cache, timeout):
    """
    Decorator is similar to memoize from flask_cache. The main difference is that,
    if cached value expires it tries to call decorated function. If the call fails
    it does not remove value from cache. Instead it resets timeout for this key.
    :param cache: cache object (for fallback use SimpleCache object only)
    :param timeout: seconds
    :return: decorated function
    """

    def memoize(f):
        @functools.wraps(f)
        def decorated_function(*args, **kwargs):

            def expired(timestamp):
                return time.time() - timestamp >= decorated_function.cache_timeout

            cache_key = decorated_function.make_cache_key(f, *args, **kwargs)

            cache_value = decorated_function.cache.get(cache_key)
            if cache_value is not None:
                value, value_timeout = cache_value
            else:
                value = None
                value_timeout = None

            if value_timeout is None:  # function value has never been cached
                value = f(*args, **kwargs)
                decorated_function.cache.set(cache_key, (value, time.time()))
            elif expired(value_timeout):
                try:
                    value = f(*args, **kwargs)
                except Exception:
                    logger.exception("Call of function {} failed".format(f))
                    decorated_function.cache.set(cache_key, (value, time.time()))

            return value

        decorated_function.cache = cache
        decorated_function.cache_timeout = timeout
        decorated_function.make_cache_key = cache._memoize_make_cache_key()

        return decorated_function

    return memoize
