import logging
from collections import defaultdict, namedtuple
from datetime import datetime, timedelta

from flask import current_app as app
from ylog.context import log_context

from jafar import clickhouse, request_cache, advisor_mongo
from jafar.utils import vanga_helpers

logger = logging.getLogger(__name__)


def get_stats_group(group_name):
    groups = [PersonalNoFallbackGroup, PersonalFallbackGroup, TotalGroup]
    return {g.name: g for g in groups}[group_name]


StatsContext = namedtuple('StatsContext', ['user', 'places_blacklist', 'packages_list',
                                           'packages_to_classnames', 'by_class_names'])


class StatsGroupGetter(object):
    def __init__(self, user, places_blacklist, packages_list, packages_to_classnames, by_class_names):
        self.context = StatsContext(user, places_blacklist, packages_list, packages_to_classnames, by_class_names)

    def get_stats(self, group_names):
        storage = []
        for group in group_names:
            storage.append(get_stats_group(group)(self.context).get())

        stats = self.merge_stats(storage)
        for (item, class_name), value in stats.iteritems():
            value['package_name'] = item
            if self.context.by_class_names and class_name:
                value['class_name'] = class_name

        return stats.values()

    @staticmethod
    def merge_stats(groups):
        result = defaultdict(dict)
        for features in groups:
            for key, value in features.iteritems():
                result[key].update(value)
        return result


class BaseStatsGroup(object):
    name = None

    def __init__(self, context):
        self.context = context

    def get(self):
        with log_context(group_name=self.name):
            logger.debug('Getting %s stats group', self.name)
            return self._get()

    def _get(self):
        raise NotImplementedError

    @staticmethod
    @request_cache.memoize()
    def _get_profile(device_id):
        return advisor_mongo.db.profile.find_one({'_id': device_id})


class PersonalNoFallbackGroup(BaseStatsGroup):
    name = 'personal_no_fallback'

    def __init__(self, context):
        super(PersonalNoFallbackGroup, self).__init__(context)
        self._prepare_places(context.places_blacklist)
        self._prepare_filter_clause()

    def _prepare_places(self, places_blacklist):
        self.places_short = []
        self.places_full = []
        for place in places_blacklist:
            if len(place) == 1:
                self.places_short.append(place)
            else:
                self.places_full.append(place)

        self.places_short = tuple(self.places_short)
        self.places_full = tuple(self.places_full)

    def _prepare_filter_clause(self):
        conditions = []

        if self.places_short:
            conditions.append('array(place) NOT IN %(places_short)s')
        if self.places_full:
            conditions.append('array(place, subplace) NOT IN %(places_full)s')
        if self.context.packages_list:
            conditions.append('item in %(packages_list)s')

        if conditions:
            self.filter_clause = 'AND ({})'.format(' AND '.join(conditions))

    def _get(self):
        result = defaultdict(dict)
        any_launch = vanga_helpers.get_any_launch(self.context.user)
        if len(any_launch) > 0:
            by_weekdays, by_hours = self.get_personal_stats(self.context.user)
        else:
            by_weekdays, by_hours = {}, {}

        for item, class_name, weekly_app_launch_count in by_weekdays:
            key = (item, class_name)
            if item not in self.context.packages_to_classnames \
                    or class_name in self.context.packages_to_classnames[item]:
                result[key]['weekly'] = dict(weekly_app_launch_count)
                result[key]['personal'] = sum(count for week, count in weekly_app_launch_count)

        for item, class_name, hourly_app_launch_count in by_hours:
            key = (item, class_name)
            if item not in self.context.packages_to_classnames \
                    or class_name in self.context.packages_to_classnames[item]:
                result[key]['hourly'] = dict(hourly_app_launch_count)

        return result

    def get_personal_stats(self, device_id):
        if self.context.by_class_names:
            query = vanga_helpers.vanga_query_personal_class_name
        else:
            query = vanga_helpers.vanga_query_personal_package_name

        params = {
            'user': device_id,
            'places_short': self.places_short,
            'places_full': self.places_full
        }

        if self.context.packages_list:
            params['packages_list'] = tuple(self.context.packages_list)

        by_weekdays = clickhouse.execute(query.format(
            db=app.config['CLICKHOUSE_DATABASE'],
            filter_clause=self.filter_clause,
            table_name=app.config['CLICKHOUSE_USAGE_WEEKLY_TABLE'],
            context='weekday'
        ), params)

        by_hours = clickhouse.execute(query.format(
            db=app.config['CLICKHOUSE_DATABASE'],
            filter_clause=self.filter_clause,
            table_name=app.config['CLICKHOUSE_USAGE_HOURLY_TABLE'],
            context='hour'
        ), params)

        if bool(by_weekdays) != bool(by_hours):
            if by_hours:
                logger.warning('"by_weekdays" is empty while "by_hours" is not')
            else:
                logger.warning('"by_hours" is empty while "by_weekdays" is not')

        return by_weekdays, by_hours


class PersonalFallbackGroup(PersonalNoFallbackGroup):
    name = 'personal_fallback'

    def _get(self):
        result = super(PersonalFallbackGroup, self)._get()
        if result:
            return result

        logger.debug('Got no personal stats.')
        if self.context.packages_list:
            logger.debug('Getting total stats for required packages.')
            return vanga_helpers.get_apps_total_stats(self.context.packages_list,
                                                      self.context.packages_to_classnames,
                                                      self.context.by_class_names)

        profile = self._get_profile(self.context.user) or {}
        if self._should_skip(profile):
            logger.debug('User is too old, return nothing.')
        else:
            logger.debug('Falling back to total stats.')
            basket = [app_['package_name'] for app_ in profile.get('installed_apps_info', [])]
            result = vanga_helpers.get_apps_total_stats(basket,
                                                        defaultdict(set),
                                                        self.context.by_class_names)

        return result

    @staticmethod
    def _should_skip(profile):
        # If date_created is None, we can't get users' apps and can't provide counters for him.
        # If date_created < date_default, user is not new. He most likely didn't have internet for
        # USAGE_STATS_SAVE_HORIZON days but does have counters on device,
        # therefore we don't update them and return nothing.
        # If len(apps) == 0, we can't give him counters because we don't know his apps
        date_created = profile.get('created_at')
        date_default = (datetime.utcnow() - timedelta(days=app.config['VANGA_NEW_USER_DURATION']))
        return date_created is None or date_created < date_default


class TotalGroup(BaseStatsGroup):
    name = 'total'

    def _get(self):
        if not self.context.packages_list:
            profile = self._get_profile(self.context.user) or {}
            basket = [app_['package_name'] for app_ in profile.get('installed_apps_info', [])]
            packages_to_classnames = defaultdict(set)
        else:
            basket = self.context.packages_list
            packages_to_classnames = self.context.packages_to_classnames

        result = vanga_helpers.get_apps_total_stats(basket,
                                                    packages_to_classnames,
                                                    self.context.by_class_names)
        return self._rename_stats(result)

    @staticmethod
    def _rename_stats(stats):
        """Stats are stored like 'personal', renaming it to 'total'"""
        for stats_item in stats.itervalues():
            for name in stats_item.keys():
                if name not in {'class_name', 'package_name', 'personal'}:
                    stats_item['total_{}'.format(name)] = stats_item.pop(name)
            stats_item['total'] = stats_item.pop('personal')

        return stats
