from collections import defaultdict

from flask import current_app as app, request

from jafar import mongo_configs
from jafar.feed import schema
from jafar.feed.views import BaseJafarView
from jafar.vanga.stats_getter import StatsGroupGetter
from jafar.web import JsonResponse
from jafar.web.exceptions import BadRequest

CLASS_NAME_SEPARATOR = '/'

class VangaView(BaseJafarView):
    experiment_config = mongo_configs.VangaExperimentConfig
    request_schema = schema.VangaSchema

    def parse_request_params(self, request_params):
        if request.method.lower() == 'post':
            self.request_schema = schema.PostVangaSchema

        return super(VangaView, self).parse_request_params(request_params)

    @staticmethod
    def _parse_place(raw_place):
        validation_dict = app.config['VANGA_BLACKLIST_PLACES']
        place = raw_place.split('.')

        if place[0] not in validation_dict or (len(place) > 1 and place[1] not in validation_dict[place[0]]):
            raise BadRequest('Unknown place %s' % place)

        return place

    @staticmethod
    def _parse_packages(packages):
        packages_to_classnames = defaultdict(set)  # Packages with class_names
        packages_list = []                # All packages
        for package in packages:
            package_info = package.split(CLASS_NAME_SEPARATOR)
            packages_list.append(package_info[0])
            if len(package_info) == 2:
                packages_to_classnames[package_info[0]].add(package_info[1])
            elif len(package_info) > 2:
                raise ValueError('Package name contains multiple separators: {}'.format(package))


        return packages_to_classnames, list(packages_list)

    def _respond(self, request_params, experiment_name):
        config = self.get_experiment_config(experiment_name)

        packages_to_classnames, packages_list = self._parse_packages(request_params.get('packages', []))
        getter = StatsGroupGetter(user=request_params['device_id'],
                                  places_blacklist=map(self._parse_place, request_params.get('places_blacklist', [])),
                                  packages_list=packages_list,
                                  packages_to_classnames=packages_to_classnames,
                                  by_class_names=(request_params.get('version', 1) > 1))
        stats = getter.get_stats(config.get('stats_groups'))

        result = {'stats': stats, 'lifetime_seconds': config['lifetime_seconds']}
        if not result['stats']:
            result['lifetime_seconds'] = config['exceptional_lifetime']
        response, errors = schema.VangaResponseSchema().dump(result)

        return JsonResponse(response)

    def post(self, request_params, experiment_name):
        return self._respond(request_params, experiment_name)

    def get(self, request_params, experiment_name):
        return self._respond(request_params, experiment_name)

