from collections import defaultdict

from httpgeobase import EARTH_REGION_ID, COUNTRY_TYPE, CITY_TYPE
from nile.api.v1 import Record

from jafar_yt.utils.helpers import get_lookuper

REDUCE_FIELDS = ['user', 'item']
STAT_FIELDS = [
    'app_install_count',
    'app_launch_count'
]

# Localized apps filter options
MIN_LOCAL_APP_INSTALLS = 50
MIN_UNIQUE_USERS = 50


class LocationStatsMapper(object):
    region_parents = {}
    region_types = {}

    def get_parents(self, region_id):
        if region_id in self.region_parents:
            return self.region_parents[region_id]
        try:
            parents = self.lookuper.parents(region_id)
        except RuntimeError:  # region_id not found
            parents = [-1]
        self.region_parents[region_id] = parents
        return parents

    def get_type(self, region_id):
        if region_id in self.region_types:
            return self.region_types[region_id]
        try:
            region_type = self.lookuper.region_by_id(region_id).type
        except RuntimeError:  # region_id not found
            region_type = -1
        self.region_types[region_id] = region_type
        return region_type

    def get_region_id(self, lat, lon, region_id):
        if lat and lon:
            try:
                return self.lookuper.region_id_by_location(lat, lon)
            except RuntimeError:
                pass
        return region_id

    def __call__(self, records):
        self.lookuper = get_lookuper()
        for record in records:
            if 'region_id' not in record:
                continue

            region_id = self.get_region_id(record.get('latitude'), record.get('longitude'), record.region_id)
            for parent_region_id in self.get_parents(region_id):
                new_row = Record(
                    item=record.item,
                    region_id=parent_region_id,
                    region_type=self.get_type(parent_region_id),
                    user=record.user,
                    **{k: record[k] for k in STAT_FIELDS}
                )
                yield new_row


def location_reducer(groups):
    for key, records in groups:
        results = {}
        region_types = {}
        unique_users = defaultdict(set)
        for record in records:
            region_id = record.region_id
            region_types[region_id] = record.region_type
            if region_id not in results:
                results[region_id] = {field: 0 for field in STAT_FIELDS}
            else:
                for field in STAT_FIELDS:
                    results[region_id][field] += record[field]
            if record['app_launch_count'] > 0:
                unique_users[region_id].add(record.user)
        try:
            total = results[EARTH_REGION_ID]
        except KeyError:
            continue
        for region_id, counters in results.iteritems():
            region_type = region_types[region_id]
            if region_type >= COUNTRY_TYPE:
                result = dict(
                    item=key.item,
                    region_id=region_id,
                    region_type=region_type,
                )
                for field in STAT_FIELDS:
                    result.update(
                        **{field + "_share": float(counters[field]) / total[field] if total[field] > 0 else 0})
                    result.update(**{field + "_total": total[field]})
                    result.update(**{field: counters[field]})
                result['unique_users'] = len(unique_users[region_id])
                yield Record.from_dict(result)


def location_stats_filter(records):
    for record in records:
        if (COUNTRY_TYPE <= record.region_type <= CITY_TYPE and
                    record.app_install_count > MIN_LOCAL_APP_INSTALLS and
                    record.unique_users > MIN_UNIQUE_USERS):
            yield Record(
                item=record.item,
                score=record.app_launch_count_share,
                region=record.region_id,
            )


def country_mapper(records):
    lookup = get_lookuper()
    for record in records:
        try:
            country_id = lookup.find_country_id(record.region)
            country = lookup.region_by_id(country_id).iso_name
            yield Record(record, country=country)
        except:
            continue

