import numpy as np
from nile.api.v1 import Record, get_records_from_file
from qb2.api.v1 import typing as qt

from jafar_yt.profile_dataframe_converter import UserProfileConverter, yt_to_mongo
from jafar_yt.utils.helpers import mapper_wrapper

"""
Contains mappers processing YT dataset tables,
filtering by required countries, adding item categories etc.
"""


class BaseInteractionsMapper(object):
    """
    Outputs user-item interactions (can be app installs,
    app launches or other types of interactions).
    """

    def __init__(self, country=None):
        self.country = country


class BaseAdvisorMongoMapper(BaseInteractionsMapper):
    """
    Processes //home/advisor/users table
    populated with records from Advisor's mongo
    """

    def check_country(self, profile):
        """
        Check that country is correctly located (`fix_country_init`)
        """
        lbs_info = profile.get('lbs_info', {})
        if lbs_info is None:
            return False
        fix_country_init = lbs_info.get('fix_country_init')
        country = lbs_info.get('country')
        country_is_valid = not fix_country_init
        return country_is_valid and country == self.country

    def __call__(self, records):
        for record in records:
            record = record.to_dict()
            try:
                record = yt_to_mongo(record)
                if self.country is None or self.check_country(record):
                    try:
                        for result in self.generate_result_rows(record):
                            yield result
                    except UserProfileConverter.IncompleteProfile:
                        continue
            except ValueError:
                pass

    def generate_result_rows(self, record):
        raise NotImplementedError


@mapper_wrapper
class AdvisorMongoInstallMapper(BaseAdvisorMongoMapper):
    schema = dict(user=qt.String, item=qt.String, install_time=qt.Integer, is_user_app=qt.Bool)

    def __init__(self, country=None, user_apps_only=True):
        self.country = country
        self.user_apps_only = user_apps_only

    def generate_result_rows(self, record):
        profile = UserProfileConverter(record)
        try:
            installs = profile.get_installs(user_apps_only=self.user_apps_only)
        except UserProfileConverter.IncompleteProfile:
            return
        for row in installs.to_list_of_dicts():
            yield Record(**row)


class AdvisorMongoLaunchMapper(BaseAdvisorMongoMapper):
    def generate_result_rows(self, record):
        if 'lbs_info' in record:
            del record['lbs_info']
        yield Record(**record)


class AdvisorMongoUserFeatureMapper(BaseAdvisorMongoMapper):
    @staticmethod
    def convert_float_to_object(array):
        """
        Converts any float column which has nans in it to
        np.object type and replaces nans with Nones, because Yt
        doesn't allow nan values.
        """
        for column, (dtype, _) in array.dtype.fields.iteritems():
            if np.issubdtype(dtype, np.float):
                nan_idx = np.isnan(array[column])
                if nan_idx.any():
                    array = array.replace_column(array[column].astype(np.object), column)
                    array[column][nan_idx] = None
        return array

    def generate_result_rows(self, record):
        try:
            df = UserProfileConverter(record).get_user_features()
        except UserProfileConverter.IncompleteProfile:
            return
        df = self.convert_float_to_object(df)
        user_features = df.to_list_of_dicts()[0]
        yield Record(**user_features)


class GeneralMetrikaInstallMapper(BaseInteractionsMapper):
    """
    Processes //home/advisor/event_identity_installs
    """

    def record_is_valid(self, record):
        if record['country'] == 'US' and record['app'] == 'com.samsung.mdl.radio':
            # some metrika bug or else, but there are LOTS of "installs" of this one in US
            return False
        return True

    def __call__(self, records):
        for record in records:
            if self.record_is_valid(record) and record.get('country') == self.country:
                yield Record(
                    item=record['app'],
                    user=record['user'],
                    install_time=record['install_time']
                )


class ConversionsMapper(BaseInteractionsMapper):
    """
    Processes //home/advisor/sheeva/aggregated
    """

    def filter(self, record):
        raise NotImplementedError

    def __call__(self, records):
        for record in records:
            if self.filter(record) and record.get('country') == self.country:
                n_installs = record['install']
                yield Record(
                    item=record['item'],
                    user=record['user'],
                    view_count=record['view_count'],
                    placement=record['placement'],
                    click=record['click'],
                    value=(1 if n_installs > 0 else 0)
                )


class ConversionsNoPromoMapper(ConversionsMapper):
    def filter(self, record):
        return record['promo'] == 0


class ConversionsPromoMapper(ConversionsMapper):
    def filter(self, record):
        return record['promo'] == 1


# ==== filters and helper functions ==== #


def fix_invalid_timestamps(records):
    for record in records:
        timestamp_length = np.ceil(np.log10(record['install_time'])).astype(int)
        if timestamp_length == 10:
            install_time = record['install_time']
        elif timestamp_length == 13:  # millisecond timestamp
            install_time = record['install_time'] // 1000
        else:
            continue
        yield Record(record, install_time=install_time)


def unroll_update_counts(groups):
    threshold = [r.threshold for r in get_records_from_file('threshold')][0]
    for key, records in groups:
        records = list(records)
        # NOTE: not using records.count() because it depletes iterator
        if len(records) > threshold:
            for record in records:
                yield record
