import json
from uuid import UUID

from nile.api.v1 import Record

REC_EVENTS = ['rec_view', 'rec_install', 'rec_click']


class MapperEvents(object):

    def __init__(self, ignore):
        # need to pass this flag from updater to mapper and remember it in temporary YT table
        self.ignore = ignore

    @staticmethod
    def get_country_from_impression_id(impression_id):
        impression_id_parts = impression_id.split(':')
        if len(impression_id_parts) < 10:
            return None
        else:
            country = impression_id_parts[9]
            return country.split(';')[0]

    def __call__(self, records):
        for record in records:
            if record.get('EventName') in REC_EVENTS:
                try:
                    event_value = json.loads(record['EventValue'])
                except ValueError:
                    continue

                package_name = event_value.get('package_name')
                impression_id = event_value.get('impression_id')
                placement = event_value.get('placement')
                try:
                    user = str(UUID(record['DeviceID']))
                except ValueError:
                    user = None

                if not all((package_name, impression_id, placement, user)):
                    continue

                offer_id = event_value.get('offer_id')
                yield Record(
                    event_name=record['EventName'],
                    user=user,
                    item=package_name,
                    timestamp=int(record['StartTimestamp']),
                    ignore=self.ignore,
                    country=self.get_country_from_impression_id(impression_id),
                    promo=(1 if offer_id and offer_id != 'empty' else 0),
                    placement=placement
                )


class TargetReducer(object):
    """
    This reducer calculates 'target' column at YT
    """

    @staticmethod
    def targets_from_events(events, event_name):
        """
        this code goes through list of event-timestamp pairs and returns list of 0/1 for each rec_view event
        :param events:
        :param event_name:
        :return:
        """

        targets = []
        last_view_event = 0
        last_target_event = 0
        for (event, timestamp) in events:
            if event == 'rec_view':
                if last_view_event > 0:  # view after previous view
                    targets.append(0)
                    last_view_event = timestamp
                    last_target_event = 0
                elif timestamp - last_target_event <= 3:  # same time
                    targets.append(1)
                    last_target_event = last_view_event = 0
                else:
                    last_view_event = timestamp
                    last_target_event = 0
            elif event == event_name:
                if timestamp - last_view_event <= 7200:  # target event caused by view
                    targets.append(1)
                    last_view_event = 0
                    last_target_event = 0
                elif last_view_event > 0:  # target event after view, but after long time - no connection
                    targets.append(0)
                    last_view_event = 0
                    last_target_event = timestamp
                else:
                    last_view_event = 0
                    last_target_event = timestamp

        if last_view_event > 0:  # last view event
            targets.append(0)

        return targets

    def __call__(self, groups):
        for result, records in groups:
            rows = sorted(set(records), key=(lambda row: row['timestamp']))
            events = map(lambda row: (row['event_name'], row['timestamp']), rows)
            view_rows = [row for row in rows if row['event_name'] == 'rec_view']
            view_ignore = map(lambda row: row['ignore'], view_rows)

            # for each view check if it caused a click
            targets_clicks = self.targets_from_events(events, 'rec_click')
            assert len(view_rows) == len(targets_clicks) and len(view_ignore) == len(targets_clicks)

            # for each view check if it caused an install
            targets_installs = self.targets_from_events(events, 'rec_install')
            assert len(view_rows) == len(targets_installs) and len(view_ignore) == len(targets_installs)

            tmp = []

            for row, ignore, click, install in zip(view_rows, view_ignore, targets_clicks, targets_installs):
                if ignore == 0:
                    tmp.append(Record(result,
                                      country=row['country'],
                                      timestamp=row['timestamp'],
                                      promo=row['promo'],
                                      placement=row['placement'],
                                      click=click,
                                      install=install))
            tmp = list(sorted(set(tmp), key=(lambda x: x['timestamp'])))
            for i, record in enumerate(tmp):
                yield Record(record, view_count=i)

