import json
from collections import defaultdict
from datetime import datetime, timedelta
from uuid import UUID

from nile.api.v1 import Record

EVENT_NAMES = [
    'rec_view',
    'app_launch',
    'App_install'
]
REDUCE_FIELDS = ['user', 'item']
STAT_FIELDS = [
    'rec_view_count',
    'app_launch_count',
    'app_install_count'
]
NESTED_STATS = [
    'hourly',
    'weekly'
]


def usage_stats_mapper(records):
    for record in records:
        try:
            event_value = json.loads(record['EventValue'])
            event_name = record['EventName']
        except ValueError:
            continue

        # sometimes 'app_launch' event means move or delete
        # see: https://wiki.yandex-team.ru/lnchr/Metrics/#zapuskprilozhenijj
        action = event_value.get('action', 'run')
        item = event_value.get('package_name') or event_value.get('packageName')
        if (event_name == 'app_launch' and action != 'run') or not item:
            continue

        try:
            latitude = float(record['Latitude'])
            longitude = float(record['Longitude'])
        except (KeyError, ValueError, TypeError):
            latitude = None
            longitude = None

        dt = datetime.strptime(record['StartTime'], '%Y-%m-%d %H:%M:%S')
        dt += timedelta(0, int(record['StartTimeZone'] or 0))

        result = Record(
            user=str(UUID(record['DeviceID'])),
            item=item,
            event_name=event_name,
            date=record['StartDate'],
            region_id=int(record.get('RegionID') or 0),
            latitude=latitude,
            longitude=longitude,
            update_time=record['ReceiveTimestamp'],
            weekday=dt.weekday(),
            hour=dt.hour
        )
        yield result


def usage_stats_reducer(groups):
    for result, records in groups:
        result = result.to_dict()
        result.update({field: 0 for field in STAT_FIELDS})
        result.update({field: defaultdict(lambda: 0) for field in NESTED_STATS})

        update_time = 0
        for record in records:
            update_time = max(record.update_time, update_time)

            if record.event_name == 'rec_view':
                result['rec_view_count'] += 1
            if record.event_name == 'App_install':
                result['app_install_count'] += 1
            if record.event_name == 'app_launch':
                result['app_launch_count'] += 1
                result['hourly'][str(record.hour)] += 1
                result['weekly'][str(record.weekday)] += 1

            if record.get('region_id'):
                result['region_id'] = record.region_id
            if record.get('latitude') and record.get('longitude'):
                result['latitude'] = record.latitude
                result['longitude'] = record.longitude
        result['update_time'] = update_time
        yield Record.from_dict(result)


def usage_stats_mapper_ch(records):
    for record in records:
        try:
            device_id = UUID(record['DeviceID'])
            event_value = json.loads(record['EventValue'])
            event_name = record['EventName']
        except (KeyError, ValueError, TypeError):
            continue

        # sometimes 'app_launch' event means move or delete
        # see: https://wiki.yandex-team.ru/lnchr/Metrics/#zapuskprilozhenijj
        action = event_value.get('action', 'run')
        item = event_value.get('package_name') or event_value.get('packageName')
        class_name = event_value.get('className', '')
        if (event_name == 'app_launch' and action != 'run') or not item:
            continue

        try:
            latitude = float(record['Latitude'])
            longitude = float(record['Longitude'])
        except (KeyError, ValueError, TypeError):
            latitude = None
            longitude = None

        result = Record(
            receive_date=record['ReceiveDate'],
            user=device_id.hex,
            item=item,
            class_name=class_name,
            event_name=event_name,
            event_value=record['EventValue'],
            event_datetime=record['EventDateTime'],
            latitude=latitude,
            longitude=longitude
        )
        yield result
