#! /usr/bin/env python
import logging
import argparse
import os
import requests

import numpy as np
from PIL import Image
from jafar.admin.models.wallpapers import WallpaperCategories, WallpaperImages, WallpaperColors, get_image_metadata
from jafar.admin.models.base import S3Image
from jafar import s3_client
from jafar import create_app

logger = logging.getLogger(__name__)

DOWNLOAD_PATH = '.'
OTHER_CATEGORY_NAME = 'other'
COLORS = {
    'gray': '#dadada',
    'yellow': '#ffbb5a',
    'green': '#59cd9c',
    'blue': '#3779bc',
    'red': '#ff6665',
    'pink': '#ea65ff',
    'black': '#403e3e',
}
COLOR_RANGES = {
    'yellow': (15, 70),
    'green': (70, 160),
    'blue': (160, 275),
    'pink': (275, 335),
    'red': (335, 15),
}


def is_black(color):
    h, s, v = color
    if s < 0.4 and v < 0.4:
        return True
    return False


def is_gray(color):
    h, s, v = color
    if s < 0.1 and v > 0.7:
        return True
    return False


def rgb_to_hsv(color):
    r, g, b = (float(c) / 255 for c in color)
    cmax = max((r, g, b))
    cmin = min((r, g, b))
    delta = cmax - cmin
    if delta == 0:
        h = 0
    elif cmax == r:
        h = 60 * (((g - b) / delta) % 6)
    elif cmax == g:
        h = 60 * ((b - r) / delta + 2)
    elif cmax == b:
        h = 60 * ((r - g) / delta + 4)
    s = 0 if cmax == 0 else delta / cmax
    v = cmax
    return (h, s, v)


def get_image_color(image):
    THRESHOLD = 0.15
    SIZE = [256, 256]

    image = Image.open(image)
    image = image.resize(SIZE)
    image = np.array(image)
    image = np.apply_along_axis(rgb_to_hsv, 2, image)

    mask_black = np.apply_along_axis(is_black, 2, image)
    mask_gray = np.apply_along_axis(is_gray, 2, image)

    if float(mask_black.sum()) / np.prod(SIZE) > 0.7:
        return 'black'
    if float(mask_gray.sum() + mask_black.sum()) / np.prod(SIZE) > 0.7:
        return 'gray'

    mask = mask_black | mask_gray

    hues = image[:, :, 0][~mask]
    ranges = sorted(COLOR_RANGES.values())
    bins = [0] + [range_[0] for range_ in ranges] + [360]
    hist, _ = np.histogram(hues.flatten(), bins)
    hist[-1] += hist[0]  # Wrap red
    hist = hist[1:]      #
    hist = [np.true_divide(n, hues.shape[0]) for n in hist]

    proper_colors = [(name, color) for name, color in COLORS.items() if name not in ['gray', 'black']]
    proper_colors = sorted(proper_colors, key=lambda k: COLOR_RANGES.get(k[0], (-1, -1)))

    n_candidates = sum([c > THRESHOLD for c in hist])
    if 0 < n_candidates <= 2:
        best = proper_colors[np.argmax(hist)][0]
        return best
    return 'other'


def save_file(data, filename):
    metadata = get_image_metadata(data)
    data.seek(0)
    key_name = 'wallpapers/images/' + filename
    s3_key = s3_client.save(data=data.read(), key_name=key_name, metadata=metadata)
    return s3_key.name


# def download_wallpaper(url):
#     file_name = url.split('/')[-1]
#     path = os.path.join(DOWNLOAD_PATH, file_name)
#     image_data = requests.get(url).content
#     with open(path, 'wb') as f:
#         f.write(image_data)


def upload_wallpapers(category_name, images):
    logger.info('Uploading category {}'.format(category_name))
    colors = {color.id_: color for color in WallpaperColors.objects()}
    category = WallpaperCategories.objects(name=category_name).first()
    if not category:
        category = WallpaperCategories(name=category_name, images=[], use_in_feed=False, use_in_autochange=False)

    for image in images:
        image_color = colors.get(get_image_color(image))
        image_colors = [image_color] if image_color else []
        with open(image, 'rb') as image_file:
            image_filename = image.split('/')[-1]
            name = save_file(image_file, image_filename)
        s3_image = S3Image(key=name)
        image_document = WallpaperImages(image=s3_image, colors=image_colors)
        category.images.append(image_document)
    category.save()


def upload_from_directory(images_dir, offset=None, limit=None):
    category_paths = [os.path.join(images_dir, category_filename)
                      for category_filename in os.listdir(images_dir)]

    categories_images = {}
    for category_path in category_paths:
        try:
            category_path_contents = os.listdir(category_path)
        except OSError:
            continue
        category_folder = os.path.basename(category_path)
        categories_images[category_folder] = []
        for image_filename in category_path_contents:
            image_path = os.path.join(category_path, image_filename)
            categories_images[category_folder].append(image_path)

        begin = offset or 0
        end = begin  + limit if limit else None
        categories_images[category_folder] = categories_images[category_folder][begin:end]

    for category, images in categories_images.items():
        upload_wallpapers(category, images)


parser = argparse.ArgumentParser(description='Upload wallpapers to mongo')
parser.add_argument('in_dir', help='Input directory', metavar='in-dir')
parser.add_argument('--offset', type=int)
parser.add_argument('--limit', type=int)

if __name__ == '__main__':
    args = parser.parse_args()
    app = create_app()
    with app.app_context():
        upload_from_directory(args.in_dir, args.offset, args.limit)
